<?php

namespace App\Http\Controllers;

use App\Models\AccountType;
use App\Models\Menu;
use App\Models\Organisation;
use App\Models\Transaction;
use App\Services\PermissionService;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    protected $permissionService;
    protected $menuId;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;

        $this->menuId = Menu::where('route', 'transactions.index')->value('id');
    }

    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Transaction Records.');
        }
        return view('auth.transactions.index');
    }

    public function getTransactionDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Transaction Records.');
        }

        $search = $request->input('search');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'transaction_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = Transaction::with('organisation', 'accountType')
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('transaction_reference', 'like', "%{$search}%")
                        ->orWhere('transaction_amount', 'like', "%{$search}%")
                        ->orWhere('transaction_date', 'like', "%{$search}%")
                        ->orWhere('transaction_narration', 'like', "%{$search}%")
                        ->orWhere('dr_cr', 'like', "%{$search}%")
                        ->orWhereHas(
                            'organisation',
                            fn($q) =>
                            $q->where('organisation_Name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'accountType',
                            fn($q) =>
                            $q->where('account_type_Name', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Transaction Records.');
        }
        $organisations = Organisation::all();
        $accountTypes = AccountType::all();
        return view('auth.transactions.create', compact('organisations', 'accountTypes'));
    }

    public function store(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Transaction Records.');
        }
        // return $request;
        $validated = $request->validate([
            //the mention fields will have data in $validated , rest in $REQUEST
            'organisation_ID'       => 'required|integer|exists:organisations,organisation_ID',
            // 'accountType_ID'        => 'nullable|integer|exists:account_types,account_type_ID',
            'transaction_reference' => 'nullable|string|max:255',
            'transaction_type'      => 'required|in:income,expense',
            'transaction_amount'    => 'required|numeric|min:0',
            'transaction_date'      => 'required|date',
            'dr_cr'                 => 'nullable|in:Dr,Cr',
            'transaction_narration' => 'required|string',
        ]);

        $dr_cr = $validated['transaction_type'] === 'expense' ? 'Dr' : 'Cr';

        Transaction::create([
            'organisation_id'        => $validated['organisation_ID'] ?? null,
            'account_type_id'        => $request->accountType_ID ?? null,
            'user_id'                => auth()->id(),
            'transaction_reference'  => $validated['transaction_reference'] ?? null,
            'transaction_type'       => $validated['transaction_type'] ?? null,
            'transaction_amount'     => $validated['transaction_amount'] ?? null,
            'transaction_date'       => $validated['transaction_date'] ?? null,
            'dr_cr'                  =>  $dr_cr ?? null,
            'transaction_narration'  => $validated['transaction_narration'] ?? null,
        ]);

        return redirect()->route('transactions.index')->with('success', 'Transaction recorded successfully!');
    }

    public function show(string $id)
    {
        //
    }

    public function eachDetail($id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Transaction Records.');
        }
        $id = base64_decode($id);
        // return $id;
        $organisations = Organisation::all();
        $accountTypes = AccountType::all();
        $transaction = Transaction::where('transaction_id', $id)->first();
        return view('auth.transactions.detail', compact('organisations', 'accountTypes', 'transaction'));
    }

    public function edit(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Transaction Records.');
        }
        $id = base64_decode($id);
        $transaction = Transaction::where('transaction_id', $id)->first();
        // return $transaction;
        $organisations = Organisation::all();
        $accountTypes = AccountType::all();
        return view('auth.transactions.edit', compact('organisations', 'accountTypes', 'transaction'));
    }


    public function update(Request $request, string $id)
    {

        // return $request;
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Transaction Records.');
        }
        $id = base64_decode($id);
        $validated = $request->validate([
            //the mention fields will have data in $validated , rest in $REQUEST
            'organisation_id'       => 'required|integer|exists:organisations,organisation_ID',
            // 'accountType_ID'        => 'nullable|integer|exists:account_types,account_type_ID',
            'transaction_reference' => 'nullable|string|max:255',
            'transaction_type'      => 'required|in:income,expense',
            'transaction_amount'    => 'required|numeric|min:0',
            'transaction_date'      => 'required|date',
            'dr_cr'                 => 'nullable|in:Dr,Cr',
            'transaction_narration' => 'required|string',
        ]);

        $dr_cr = $validated['transaction_type'] === 'expense' ? 'Dr' : 'Cr';

        Transaction::create([
            'organisation_id'        => $validated['organisation_id'] ?? null,
            'account_type_id'        => $request->account_type_id ?? null,
            'user_id'                => auth()->id(),
            'transaction_reference'  => $validated['transaction_reference'] ?? null,
            'transaction_type'       => $validated['transaction_type'] ?? null,
            'transaction_amount'     => $validated['transaction_amount'] ?? null,
            'transaction_date'       => $validated['transaction_date'] ?? null,
            'dr_cr'                  =>  $dr_cr ?? null,
            'transaction_narration'  => $validated['transaction_narration'] ?? null,
        ]);

        return redirect()->route('transactions.index')->with('success', 'Transaction Updated successfully!');
    }


    public function destroy(Request $request)
    {
        // return $id;
        $transaction = Transaction::findOrFail($request->id);
        $transaction->delete();

        return response()->json([
            'status'  => true,
            'message' => 'Transaction deleted successfully.'
        ]);
    }
}
