<?php

namespace App\Http\Controllers;

use App\Traits\CommonTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;


class LedgerController extends Controller
{
    protected $tables; // 👈 Declare it as a class property
    protected $journal_transcation_tbl;
    protected $organisation_id;
    use CommonTrait;

    public function __construct()
    {
        $this->tables = $this->getYearPrefixedTables();
        $this->organisation_id = Auth::user()?->organisation_id;
    }

    public function income(Request $request)
    {
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $organisation_id = $this->organisation_id;
        $ledgers = $this->getLedgerGroupedByHeadAndGroup($organisation_id, $dateFrom, $dateTo, 'income');
        return view('ledgers.income', ['ledger_summery' => $ledgers]);
    }

    public function expense(Request $request)
    {
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $organisation_id = $this->organisation_id;
        $ledgers = $this->getLedgerGroupedByHeadAndGroup($organisation_id, $dateFrom, $dateTo, 'expense');
        return view('ledgers.expense', ['ledger_summery' => $ledgers]);
    }

    public function donersLedger(Request $request){
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $organisation_id = $this->organisation_id;
        $donorsWithRecords = $this->getDonerLedgerList($organisation_id, $dateFrom, $dateTo, 'income');
        return view('ledgers.doners', ['donorsWithRecords' => $donorsWithRecords]);
    }

    public function vendorsLedger(Request $request){
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $organisation_id = $this->organisation_id;
        $vendorsWithRecords = $this->getVendorLedgerList($organisation_id, $dateFrom, $dateTo, 'expense');
        return view('ledgers.vendors', ['vendorsWithRecords' => $vendorsWithRecords]);
    }

    public function getDonorDetails(Request $request, $donor, $id = null)
    {
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $organisation_id = $this->organisation_id;

        // Decode base64 encoded donor_id if provided, otherwise use as is
        $donor_id = $id;
        if ($donor_id) {
            // Try to decode base64, if it fails, use the original value
            $decoded = base64_decode($donor_id, true);
            if ($decoded !== false && is_numeric($decoded)) {
                $donor_id = $decoded;
            }
        }

        if (!$donor_id) {
            abort(404, 'Donor ID not provided');
        }

        // Get year-prefixed table names
        $donorMappingTable = $this->tables['donor_transaction_record'];
        $journalTable = $this->tables['journal_transactions'];

        // Get donor information
        $donorRecord = DB::table('donors')
            ->where('donor_id', $donor_id)
            ->first();

        if (!$donorRecord) {
            abort(404, 'Donor not found');
        }


        // Get all transactions for this donor
        $donorRecordsQuery = DB::table("$donorMappingTable as dr")
            ->leftJoin("$journalTable as jt", 'dr.journal_id', '=', 'jt.id')
            ->leftJoin('heads as h', 'h.id', '=', 'jt.head_id')
            ->leftJoin('head_groups as h_g', 'h_g.id', '=', 'h.head_group_id')
            ->where('dr.donor_id', $donor_id)
            ->where('jt.organisation_id',  $organisation_id)
            ->when($dateFrom && $dateTo, function($q) use ($dateFrom, $dateTo) {
                $q->whereBetween('jt.transaction_date', [$dateFrom, $dateTo]);
            })
            ->orderBy('jt.transaction_date', 'asc')
            ->select(
                'dr.*',
                'jt.transaction_date',
                'jt.transaction_amount',
                'jt.transaction_narration',
                'h.name as head_name',
                'h_g.name as head_group_name',
                'jt.money_receipt_no as money_receipt_no'
            );

        $donorRecords = $donorRecordsQuery->get();
        return view('ledgers.doner_detail', ['donor' => $donorRecord, 'donorRecords' => $donorRecords]);
    }

    public function getVendorDetails(Request $request, $vendor, $id = null)
    {
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $organisation_id = $this->organisation_id;

        // Decode base64 encoded vendor_id if provided, otherwise use as is
        $vendor_id = $id;
        if ($vendor_id) {
            // Try to decode base64, if it fails, use the original value
            $decoded = base64_decode($vendor_id, true);
            if ($decoded !== false && is_numeric($decoded)) {
                $vendor_id = $decoded;
            }
        }

        if (!$vendor_id) {
            abort(404, 'Vendor ID not provided');
        }

        // Get year-prefixed table names
        $vendorMappingTable = $this->tables['vendor_transaction_record'];
        $journalTable = $this->tables['journal_transactions'];

        // Get vendor information
        $vendorRecord = DB::table('vendors')
            ->where('vendor_id', $vendor_id)
            ->first();

        if (!$vendorRecord) {
            abort(404, 'Vendor not found');
        }

        // Get vendor type
        $vendorType = $vendorRecord->vendor_type ?? 'external';

        // Get all transactions for this vendor
        $vendorRecordsQuery = DB::table("$vendorMappingTable as vr")
            ->leftJoin("$journalTable as jt", 'vr.journal_id', '=', 'jt.id')
            ->leftJoin('heads as h', 'h.id', '=', 'jt.head_id')
            ->leftJoin('head_groups as h_g', 'h_g.id', '=', 'h.head_group_id')
            ->where('vr.vendor_id', $vendor_id)
            ->where('jt.organisation_id',  $organisation_id)
            ->when($dateFrom && $dateTo, function($q) use ($dateFrom, $dateTo) {
                $q->whereBetween('vr.created_at', [$dateFrom, $dateTo]);
            })
            ->orderBy('vr.created_at', 'asc')
            ->select(
                'vr.*',
                'vr.total_amount',
                'vr.paid_amount',
                'vr.created_at',
                'jt.transaction_amount',
                'jt.transaction_narration',
                'h.name as head_name',
                'h.name as expense_head_name',
                'h_g.name as head_group_name',
                'jt.money_receipt_no as money_receipt_no'
            );

        $vendorRecords = $vendorRecordsQuery->get();
        return view('ledgers.vendor_detail', ['vendor' => $vendorRecord, 'vendorRecords' => $vendorRecords, 'vendorType' => $vendorType]);
    }

    public function getIncomeHeadDetails(Request $request, $transaction_type, $head_id = null)
    {

        $organisation_id = $this->organisation_id;
        $dateFrom = $request->input('date_from', null);
        $dateTo = $request->input('date_to', null);
        $head_id = $head_id ? base64_decode($head_id) : null;
        $transaction_type = $transaction_type;
        $ledger_details = $this->headLedgerDetails($organisation_id, $head_id, $dateFrom, $dateTo, $transaction_type);
        //dd($ledger_details);
        return view('ledgers.detail', ['ledger_head_details' => $ledger_details]);
    }

    private function getLedgerGroupedByHeadAndGroup($organisation_id = null, $dateFrom = null, $dateTo = null, $transaction_type = null )
    {
        $journalTable = $this->tables['journal_transactions'];

        // Fetch transaction details with group + head info
        $rows = DB::table("$journalTable as jt")
            ->select(
                'hg.id as group_id',
                'hg.name as group_name',
                'jt.transcation_type as type',
                'h.id as head_id',
                'h.name as head_name',
                'jt.id as transaction_id',
                'jt.transaction_date',
                'jt.transaction_amount',
                'jt.transaction_narration as narration'
            )
            ->join('head_groups as hg', 'hg.id', '=', 'jt.head_group_id')
            ->join('heads as h', 'h.id', '=', 'jt.head_id')
            ->when($organisation_id, fn($q) => $q->where('jt.organisation_id', $organisation_id))
            ->when($dateFrom && $dateTo, fn($q) => $q->whereBetween('jt.transaction_date', [$dateFrom, $dateTo]))
            ->when($transaction_type, fn($q) => $q->where('jt.transcation_type', $transaction_type))
            ->orderBy('hg.name')
            ->orderBy('h.name')
            ->orderBy('jt.transaction_date')
            ->get();

        // Group the data hierarchically
        $groups = collect();

        foreach ($rows as $r) {
            if (! $groups->has($r->group_id)) {
                $groups->put($r->group_id, (object)[
                    'head_group_id' => $r->group_id,
                    'group_name' => $r->group_name,
                    'type' => $r->type,
                    'group_total' => 0.0,
                    'heads' => collect()
                ]);
            }

            $group = $groups->get($r->group_id);

            // Head group level
            if (! $group->heads->has($r->head_id)) {
                $group->heads->put($r->head_id, (object)[
                    'head_id' => $r->head_id,
                    'head_name' => $r->head_name,
                    'head_total' => 0.0,
                    'transactions' => collect()
                ]);
            }

            $head = $group->heads->get($r->head_id);

            // Add transaction details
            $head->transactions->push((object)[
                'id' => $r->transaction_id,
                'date' => $r->transaction_date,
                'amount' => (float)$r->transaction_amount,
                'narration' => $r->narration,
            ]);

            // Update totals
            $head->head_total += (float)$r->transaction_amount;
            $group->group_total += (float)$r->transaction_amount;

            // Save back
            $group->heads->put($r->head_id, $head);
            $groups->put($r->group_id, $group);
        }

        // Split by income / expense
        $income = $groups->filter(fn($g) => $g->type === 'income')->values();
        $expense = $groups->filter(fn($g) => $g->type === 'expense')->values();

        if ($transaction_type === 'income') {
            return $income;
        } elseif ($transaction_type === 'expense') {
            return $expense;
        } else {
            return (object)[
                'income' => $income,
                'expense' => $expense
            ];
        }
    }

    private function headLedgerDetails($organisation_id = null, $id = null, $dateFrom = null, $dateTo = null, $transaction_type = null)
    {
        $journalTable = $this->tables['journal_transactions'];

        // ✅ Fetch transaction details
        $rows = DB::table("$journalTable as jt")
            ->select(
                'hg.id as group_id',
                'hg.name as group_name',
                'jt.transcation_type as type',
                'h.id as head_id',
                'h.name as head_name',
                'jt.id as transaction_id',
                'jt.transaction_date',
                'jt.transaction_amount',
                'jt.transaction_narration as narration',
                'jt.payment_type_id'
            )
            ->join('head_groups as hg', 'hg.id', '=', 'jt.head_group_id')
            ->join('heads as h', 'h.id', '=', 'jt.head_id')
            ->when($organisation_id, fn($q) => $q->where('jt.organisation_id', $organisation_id))
            ->when($id, fn($q) => $q->where('jt.head_id', $id))
            ->when($dateFrom && $dateTo, fn($q) => $q->whereBetween('jt.transaction_date', [$dateFrom, $dateTo]))
            ->when($transaction_type, fn($q) => $q->where('jt.transcation_type', $transaction_type))
            ->orderBy('hg.name')
            ->orderBy('h.name')
            ->orderBy('jt.transaction_date')
            ->get();

        // ✅ Group data by head_id
        $heads = collect();

        foreach ($rows as $r) {
            if (!$heads->has($r->head_id)) {
                $heads->put($r->head_id, (object) [
                    'head_id' => $r->head_id,
                    'head_name' => $r->head_name,
                    'group_id' => $r->group_id,
                    'group_name' => $r->group_name,
                    'type' => $r->type,
                    'head_total' => 0.0,
                    'transaction_count' => 0,
                    'transactions' => collect(),
                    'cash_transactions' => collect(),   // 🟢 New
                    'bank_transactions' => collect(),   // 🟢 New
                    'cash_total' => 0.0,                // 🟢 New
                    'bank_total' => 0.0,                // 🟢 New
                ]);
            }

            $head = $heads->get($r->head_id);

            // ✅ Get payment type name safely
            $payment_type = $this->getPayment_TypeName_Idwise($r->payment_type_id);
            $payment_type_name = $payment_type->name ?? 'N/A';

            $transaction = (object) [
                'trans_id' => $r->transaction_id,
                'date' => $r->transaction_date,
                'amount' => (float) $r->transaction_amount,
                'narration' => $r->narration,
                'payment_type' => $payment_type_name,
                'payment_type_id' => $r->payment_type_id,
            ];

            // ✅ Add transaction to main list
            $head->transactions->push($transaction);

            // ✅ Separate by payment type
            if ($r->payment_type_id == 1) {
                $head->cash_transactions->push($transaction);
                $head->cash_total += (float) $r->transaction_amount;
            } else {
                $head->bank_transactions->push($transaction);
                $head->bank_total += (float) $r->transaction_amount;
            }

            // ✅ Update totals and counts
            $head->head_total += (float) $r->transaction_amount;
            $head->transaction_count++;

            $heads->put($r->head_id, $head);
        }

        // ✅ Split by type (income / expense)
        $income = $heads->filter(fn($h) => $h->type === 'income')->values();
        $expense = $heads->filter(fn($h) => $h->type === 'expense')->values();

        // ✅ Return based on type
        if ($transaction_type === 'income') {
            return $income;
        } elseif ($transaction_type === 'expense') {
            return $expense;
        }

        // ✅ Default (return both)
        return (object) [
            'income' => $income,
            'expense' => $expense,
        ];
    }

    private function getDonerLedgerList ($organisation_id = null, $dateFrom = null, $dateTo = null, $transaction_type = null ){
        $journalTable = $this->tables['journal_transactions'];
        // Fetch transaction details with group + head info
        $rows = DB::table("$journalTable as jt")
        ->select(
                'hg.id as group_id',
                'hg.name as group_name',
                'jt.transcation_type as type',
                'h.id as head_id',
                'h.name as head_name',
                'jt.id as transaction_id',
                'jt.transaction_date',
                'jt.transaction_amount',
                'jt.donor_id',
                'donors.donor_name',
                'donors.donor_id',
                'jt.transaction_narration as narration'
            )
            ->join('head_groups as hg', 'hg.id', '=', 'jt.head_group_id')
            ->join('heads as h', 'h.id', '=', 'jt.head_id')
            ->join('donors', 'donors.donor_id', '=', 'jt.donor_id')
            ->when($organisation_id, fn($q) => $q->where('jt.organisation_id', $organisation_id))
            ->when($dateFrom && $dateTo, fn($q) => $q->whereBetween('jt.transaction_date', [$dateFrom, $dateTo]))
            ->when($transaction_type, fn($q) => $q->where('jt.transcation_type', $transaction_type))
            ->whereNotNull('jt.donor_id')
            ->where('donors.donor_status', 'active')
            ->orderBy('donors.donor_name')
            ->orderBy('hg.name')
            ->orderBy('h.name')
            ->orderBy('jt.transaction_date')
            ->get();

        if(!empty($rows)){
            // Group by donor_id and sum the total amount
            $groupedDonors = collect();

            foreach ($rows as $row) {
                $donorId = $row->donor_id;

                if (!$groupedDonors->has($donorId)) {
                    $groupedDonors->put($donorId, (object)[
                        'donor_id' => $row->donor_id,
                        'donor_name' => $row->donor_name,
                        'transaction_amount' => 0.0,
                        'transaction_count' => 0
                    ]);
                }

                $donor = $groupedDonors->get($donorId);
                $donor->transaction_amount += (float)$row->transaction_amount;
                $donor->transaction_count++;

                $groupedDonors->put($donorId, $donor);
            }

            return $groupedDonors->values();
        }else{
            return collect();
        }
    }


    private function getVendorLedgerList($organisation_id = null, $dateFrom = null, $dateTo = null, $transaction_type = null){
        $journalTable = $this->tables['journal_transactions'];
        $rows = DB::table("$journalTable as jt")
        ->select(
                'hg.id as group_id',
                'hg.name as group_name',
                'jt.transcation_type as type',
                'h.id as head_id',
                'h.name as head_name',
                'jt.id as transaction_id',
                'jt.transaction_date',
                'jt.transaction_amount',
                'jt.vendor_id',
                'vendors.vendor_name',
                'vendors.vendor_id',
                'jt.transaction_narration as narration'
            )
            ->join('head_groups as hg', 'hg.id', '=', 'jt.head_group_id')
            ->join('heads as h', 'h.id', '=', 'jt.head_id')
            ->join('vendors', 'vendors.vendor_id', '=', 'jt.vendor_id')
            ->when($organisation_id, fn($q) => $q->where('jt.organisation_id', $organisation_id))
            ->when($dateFrom && $dateTo, fn($q) => $q->whereBetween('jt.transaction_date', [$dateFrom, $dateTo]))
            ->when($transaction_type, fn($q) => $q->where('jt.transcation_type', $transaction_type))
            ->whereNotNull('jt.vendor_id')
            ->where('vendors.vendor_status', 'active')
            ->orderBy('vendors.vendor_name')
            ->orderBy('hg.name')
            ->orderBy('h.name')
            ->orderBy('jt.transaction_date')
            ->get();

        if(!empty($rows)){
            // Group by vendor_id and sum the total amount
            $groupedVendors = collect();

            foreach ($rows as $row) {
                $vendorId = $row->vendor_id;

                if (!$groupedVendors->has($vendorId)) {
                    $groupedVendors->put($vendorId, (object)[
                        'vendor_id' => $row->vendor_id,
                        'vendor_name' => $row->vendor_name,
                        'transaction_amount' => 0.0,
                        'transaction_count' => 0
                    ]);
                }

                $vendor = $groupedVendors->get($vendorId);
                $vendor->transaction_amount += (float)$row->transaction_amount;
                $vendor->transaction_count++;
                $groupedVendors->put($vendorId, $vendor);
            }

            return $groupedVendors->values();
        }else{
            return collect();
        }
    }
}
