<?php

namespace App\Http\Controllers;

use App\Models\AccountingYear;
use App\Models\Menu;
use App\Models\Organisation;
use App\Services\PermissionService;
use Illuminate\Http\Request;

class AccountingYearController extends Controller
{

    protected $permissionService;
    protected $menuId;



    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;

        $this->menuId = Menu::where('route', 'accounting_years.index')->value('id');
    }

    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Accounting Years.');
        }
        return view('auth.accountingYears.index');
    }

    public function getAccountingYearDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Accounting Years.');
        }

        $search = $request->input('search');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'accounting_year_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = AccountingYear::query()
            ->when(
                $search,
                fn($q) =>
                $q->where('accounting_year_financial', 'like', "%$search%")
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Accounting Years.');
        }
        $activeYears = AccountingYear::where('status', 'active')
            ->pluck('accounting_year_financial')
            ->toArray();

        $inactiveYears = AccountingYear::where('status', 'inactive')
            ->pluck('accounting_year_financial')
            ->toArray();
        $organisations = Organisation::where('status', 'active')->get();
        return view('auth.accountingYears.create', compact('activeYears', 'inactiveYears', 'organisations'));
    }

    public function store(Request $request)
    {
        // return $request;
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Accounting Years.');
        }

        $organisation_id = $request->organisation_id;
        $range = $request->accounting_year_financial;

        AccountingYear::where('is_current', 1)->update(['is_current' => 0]);

        AccountingYear::updateOrCreate(
            [
                'accounting_year_financial' => $range,
            ],
            [
                'accounting_year_financial' => $range,
                'is_current' => 1,
                'status' => 'active'
            ]
        );

        Organisation::where('is_current', 1)->update(['is_current' => 0]);

        Organisation::where('organisation_id', $organisation_id)
            ->update([
                'is_current' => 1,
                'status' => 'active',
            ]);
        return redirect()->route('accounting_years.index')
            ->with('success', 'Accounting year added successfully. Organisation and Accounting Year set successfully !!!');
    }



    public function show(string $id)
    {
        //
    }

    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
    public function status($id)
    {
        if (!$this->permissionService->hasPermission(1, 'x')) {
            abort(403, 'You do not have read access to financial years.');
        }
        $year = AccountingYear::findOrFail($id);
        if ($year->is_current == 1 && $year->status === 'active') {
            return response()->json([
                'success' => false,
                'message' => 'The current financial year cannot be deactivated.',
            ], 400);
        }
        $year->status = $year->status === 'active' ? 'inactive' : 'active';
        $year->save();
        return response()->json([
            'success' => true,
            'status' => $year->status,
        ]);
    }
}
