<?php

namespace App\Http\Controllers;

use App\Models\TransactionIncome;
use App\Models\TransactionExpense;
use App\Models\AcademicYear;
use App\Models\IncomeHead;
use App\Models\ExpenseHead;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use App\Models\Student;
use App\Models\Staff;
use App\Models\Vendor;

use Str;

class TransactionController extends Controller
{


    public function index(Request $request)
    {
        $academic_years = AcademicYear::where('status', 1)->get();
        $active_academic_year = AcademicYear::where('status', 1)->firstOrFail();

        $income_heads = IncomeHead::where('id','<>', 1)->where('status', 1)->get();
        $expense_heads = ExpenseHead::where('id','<>', 1)->where('status', 1)->get();
        $perPage = 25;

        $financialYearId = $active_academic_year->id;

        /* ================= FILTER INPUTS ================= */
        $financialYearId = $request->get('financial_year_id');
        $type            = $request->get('type'); // income | expense
        $search          = $request->get('q');
        $monthInput      = $request->get('month'); // YYYY-MM
        $transactionDate = $request->get('transaction_date'); // YYYY-MM-DD

        /* ================= MONTH PARSE ================= */
        $filterYear  = null;
        $filterMonth = null;

        if ($monthInput) {
            [$filterYear, $filterMonth] = explode('-', $monthInput);
        }

        /* ================= INCOME QUERY ================= */
        $income = DB::table('transaction_income as ti')
            ->join('income_heads as ih', 'ih.id', '=', 'ti.income_head_id')
            ->select(
                'ti.id',
                'ti.transaction_id',
                'ti.transaction_ref_id',
                'ti.transaction_amount',
                'ti.payment_method',
                'ti.transaction_date',
                DB::raw("'income' as type"),
                'ih.name as income_head_name',
                'ih.id as income_head_id',
                //DB::raw('NULL as expense_head_name'),
                //DB::raw('NULL as expense_head_id')
            )
            ->when($financialYearId, fn ($q) =>
                $q->where('ti.financial_year_id', $financialYearId)
            )
            ->when($type == 'income' && $filterYear && $filterMonth, fn ($q) =>
                $q->whereYear('ti.transaction_date', $filterYear)
                ->whereMonth('ti.transaction_date', $filterMonth)
            )
            ->when($type == 'income' && $transactionDate, fn ($q) =>
                $q->whereDate('ti.transaction_date', $transactionDate)
            )
            ->when($type == 'income' && $search, fn ($q) =>
                $q->where(function ($qq) use ($search) {
                    $qq->where('ti.transaction_id', 'like', "%{$search}%")
                    ->orWhere('ti.transaction_ref_id', 'like', "%{$search}%")
                    ->orWhere('ih.name', 'like', "%{$search}%");
                })
            )
            ->orderBy('transaction_date', 'desc')
            ->paginate($perPage, ['*'], 'income_history');

        /* ================= EXPENSE QUERY ================= */
        $expense = DB::table('transaction_expense as te')
            ->join('expense_heads as eh', 'eh.id', '=', 'te.expense_head_id')
            ->select(
                'te.id',
                'te.transaction_id',
                'te.transaction_ref_id',
                'te.transaction_amount',
                'te.payment_method',
                'te.transaction_date',
                DB::raw("'expense' as type"),
                //DB::raw('NULL as income_head_name'),
                //DB::raw('NULL as income_head_id'),
                'eh.name as expense_head_name',
                'eh.id as expense_head_id',
            )
            ->when($financialYearId, fn ($q) =>
                $q->where('te.financial_year_id', $financialYearId)
            )
            ->when($type == 'expense' && $filterYear && $filterMonth, fn ($q) =>
                $q->whereYear('te.transaction_date', $filterYear)
                ->whereMonth('te.transaction_date', $filterMonth)
            )
            ->when($type == 'expense' && $transactionDate, fn ($q) =>
                $q->whereDate('te.transaction_date', $transactionDate)
            )
            ->when($type == 'expense' && $search, fn ($q) =>
                $q->where(function ($qq) use ($search) {
                    $qq->where('te.transaction_id', 'like', "%{$search}%")
                    ->orWhere('te.transaction_ref_id', 'like', "%{$search}%")
                    ->orWhere('eh.name', 'like', "%{$search}%");
                })
            )
            ->orderBy('transaction_date', 'desc')
            ->paginate($perPage, ['*'], 'expense_history');


        /* ================= TYPE FILTER ================= */
        /*if ($type === 'income') {
            $transactions = $income
                ->orderBy('transaction_date', 'desc')
                ->paginate($perPage);

        } elseif ($type === 'expense') {
            $transactions = $expense
                ->orderBy('transaction_date', 'desc')
                ->paginate($perPage);

        } else {
            $transactions = $income
                ->unionAll($expense)
                ->orderBy('transaction_date', 'desc')
                ->paginate($perPage);
        }*/

        // echo '<pre>';
        // print_r($transactions->toArray());die;
        return view('transactions.index', compact(
            'income',
            'expense',
            'academic_years',
            'active_academic_year',
            'income_heads',
            'expense_heads'
        ));
    }


    public function storeIncome(Request $request)
    {
        try {
            $validated = $request->validate([
                'transaction_ref_id'   => 'required|string|max:100',
                'transaction_amount'   => 'required|numeric|min:1',
                'payment_method'       => 'required|string',
                'transaction_date'     => 'required|date',

                'income_head_id'       => 'required|exists:income_heads,id',

                // 'source_type'          => 'required|in:student,guest,other',

                // 'source_id' => [
                //     'nullable',
                //     Rule::requiredIf(fn () =>
                //         $request->source_type === 'student'
                //     ),
                // ],

                // 'source_name' => [
                //     'nullable',
                //     Rule::requiredIf(fn () =>
                //         $request->source_type !== 'student'
                //     ),
                //     'string',
                //     'max:255',
                // ],

                'transaction_reference_photo' =>
                    'nullable|file|mimes:jpg,jpeg,png,webp,pdf|max:2048',
            ]);

            $academicYear = AcademicYear::where('status', 1)->firstOrFail();

            $referencePath = null;
            if ($request->hasFile('transaction_reference_photo')) {
                $referencePath = $request->file('transaction_reference_photo')
                    ->store('transactions/income', 'public');
            }

            $income = TransactionIncome::create([
                'transaction_id'        => strtoupper(Str::random(10)),
                'financial_year_id'     => $academicYear->id,
                'income_head_id'        => $validated['income_head_id'],

                'transaction_amount'    => $validated['transaction_amount'],
                'payment_method'        => $validated['payment_method'],
                'transaction_date'      => $validated['transaction_date'],
                'transaction_ref_id'    => $validated['transaction_ref_id'],
                'transaction_details'   => $request->transaction_details,

                // 'source_type'           => $validated['source_type'],
                // 'source_id'             => $validated['source_id'] ?? null,
                // 'source_name'           => $validated['source_name'] ?? null,

                'transaction_reference_photo' => $referencePath,
            ]);

            return response()->json([
                'status'  => true,
                'message' => 'Income added successfully',
                'data'    => $income,
            ], 200);

        } catch (\Illuminate\Validation\ValidationException $e) {

            return response()->json([
                'status' => false,
                'errors' => $e->errors(),
            ], 422);

        } catch (\Throwable $e) {

            return response()->json([
                'status'  => false,
                'message' => 'Something went wrong',
                'error'   => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function storeExpense(Request $request)
    {
        try {
            $validated = $request->validate([
                'transaction_ref_id'   => 'required|string|max:100',
                'transaction_amount'   => 'required|numeric|min:1',
                'payment_method'       => 'required|string',
                'transaction_date'     => 'required|date',

                'expense_head_id'      => 'required|exists:expense_heads,id',

                'source_type'          => 'nullable|in:staff,vendor',

                'source_id' => [
                    'nullable',
                    Rule::requiredIf(fn () =>
                        $request->source_type === 'staff'
                    ),
                ],

                'source_name' => [
                    'nullable',
                    'string',
                    'max:255',
                ],

                'transaction_reference_photo' =>
                    'nullable|file|mimes:jpg,jpeg,png,webp,pdf|max:2048',
            ]);

            $academicYear = AcademicYear::where('status', 1)->firstOrFail();

            $referencePath = null;
            if ($request->hasFile('transaction_reference_photo')) {
                $referencePath = $request->file('transaction_reference_photo')
                    ->store('transactions/expense', 'public');
            }

            $expense = TransactionExpense::create([
                'transaction_id'        => strtoupper(Str::random(10)),
                'financial_year_id'     => $academicYear->id,
                'expense_head_id'       => $validated['expense_head_id'],

                'transaction_amount'    => $validated['transaction_amount'],
                'payment_method'        => $validated['payment_method'],
                'transaction_date'      => $validated['transaction_date'],
                'transaction_ref_id'    => $validated['transaction_ref_id'],
                'transaction_details'   => $request->transaction_details,

                'source_type'           => $validated['source_type'],
                'source_id'             => $validated['source_id'] ?? null,
                'source_name'           => $validated['source_name'] ?? null,

                'transaction_reference_photo' => $referencePath,
            ]);

            return response()->json([
                'status'  => true,
                'message' => 'Expense added successfully',
                'data'    => $expense,
            ], 200);

        } catch (\Illuminate\Validation\ValidationException $e) {

            return response()->json([
                'status' => false,
                'errors' => $e->errors(),
            ], 422);

        } catch (\Throwable $e) {

            return response()->json([
                'status'  => false,
                'message' => 'Something went wrong',
                'error'   => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }


    public function show(string $type, int $id)
    {
        if ($type === 'income') {

            $transaction = DB::table('transaction_income as ti')
                ->join('income_heads as ih', 'ih.id', '=', 'ti.income_head_id')
                ->join('financial_years as ay', 'ay.id', '=', 'ti.financial_year_id')

                /* ================= STUDENT ================= */
                ->leftJoin('students as s', function ($join) {
                    $join->on('s.id', '=', 'ti.source_id')
                        ->where('ti.source_type', '=', 'student')
                        ->where('ti.income_head_id', '=', 1);
                })

                /* ================= STUDENT CLASS DETAILS ================= */
                ->leftJoin('student_class_details as scd', function ($join) {
                    $join->on('scd.student_id', '=', 's.id')
                        ->where('scd.status', '=', 1);
                })

                /* ================= CLASS ================= */
                ->leftJoin('classes as c', 'c.id', '=', 'scd.class_id')

                /* ================= SECTION (FIXED) ================= */
                ->leftJoin('sections as sec', 'sec.id', '=', 'scd.section_id')

                ->select(
                    /* TRANSACTION */
                    'ti.id',
                    'ti.transaction_id',
                    'ti.transaction_ref_id',
                    'ti.financial_year_id',
                    'ay.label as financial_year',
                    'ti.transaction_amount',
                    'ti.transaction_date',
                    'ti.transaction_details',
                    'ti.payment_method',
                    'ti.income_head_id as income_expense_head_id',
                    'ih.name as head_name',
                    'ti.transaction_reference_photo',
                    'ti.source_type',
                    'ti.source_name',

                    /* STUDENT */
                    's.name as student_name',
                    's.admission_no',

                    /* CLASS DETAILS */
                    //'scd.roll_no',
                    //'scd.admission_date',

                    /* CLASS / SECTION */
                    'c.class_name',
                    'sec.section_name'
                )
                ->where('ti.id', $id)
                ->first();

        } else {

            $transaction = DB::table('transaction_expense as te')
                ->join('expense_heads as eh', 'eh.id', '=', 'te.expense_head_id')
                ->join('financial_years as ay', 'ay.id', '=', 'te.financial_year_id')

                /* ================= STAFF (ONLY FOR SALARY) ================= */
                ->leftJoin('staffs as st', function ($join) {
                    $join->on('st.id', '=', 'te.source_id')
                        ->where('te.source_type', '=', 'staff');
                        //->where('te.expense_head_id', '=', 1);
                })

                ->leftJoin('vendors as v', function ($join) {
                    $join->on('v.id', '=', 'te.source_id')
                        ->where('te.source_type', '=', 'vendor');
                        //->where('te.expense_head_id', '=', 1);
                })

                ->select(
                    /* ================= TRANSACTION ================= */
                    'te.id',
                    'te.transaction_id',
                    'te.transaction_ref_id',
                    'te.financial_year_id',
                    'ay.label as financial_year',
                    'te.transaction_amount',
                    'te.transaction_date',
                    'te.transaction_details',
                    'te.payment_method',
                    'te.expense_head_id as income_expense_head_id',
                    'eh.name as head_name',
                    'te.transaction_reference_photo',
                    'te.source_type',
                    'te.source_name',

                    /* ================= STAFF  ================= */
                    'st.name as staff_name',
                    'st.designation',
                    'st.category',

                    /* ================= Vendor ================= */
                    'v.name as vendor_name'
                )
                ->where('te.id', $id)
                ->first();

        }

        abort_if(!$transaction, 404);

        /* ✅ FLAT RESPONSE (JS FRIENDLY) */
        return response()->json(
            array_merge(
                ['type' => $type],
                (array) $transaction
            )
        );
    }


    public function edit($type, $id)
    {
        $transaction = $type === 'income'
            ? TransactionIncome::findOrFail($id)
            : TransactionExpense::findOrFail($id);

        // dynamic attribute
        $transaction->type = $type;

        // default null
        //$transaction->source = null;

        // 🔗 Load related source dynamically
        if ($transaction->source_type === 'student' && $transaction->source_id) {

            $transaction->student = Student::query()
                ->select(
                    'students.admission_no',
                    'students.id',
                    'students.name',
                    'classes.class_name',
                    'sections.section_name'
                )
                ->leftJoin('student_class_details', function ($join) {
                    $join->on('student_class_details.student_id', '=', 'students.id')
                        ->where('student_class_details.status', 1);
                })
                ->leftJoin('classes', 'classes.id', '=', 'student_class_details.class_id')
                ->leftJoin('sections', 'sections.id', '=', 'student_class_details.section_id')
                ->where('students.id', $transaction->source_id)
                ->first();
        }

        if ($transaction->source_type === 'staff' && $transaction->source_id) {
            $transaction->staff = Staff::select('id', 'name', 'category')
                ->find($transaction->source_id);
        } else if ($transaction->source_type === 'vendor' && $transaction->source_id) {
            $transaction->vendor = Vendor::select('id', 'name')
                ->find($transaction->source_id);
        }

        return response()->json($transaction);
    }




    public function updateIncome(Request $request, $id)
    {
        $income = TransactionIncome::findOrFail($id);

        $request->validate([
            'income_head_id'        => 'required|exists:income_heads,id',
            'transaction_amount'   => 'required|numeric|min:1',
            'payment_method'       => 'required|string',
            'transaction_date'     => 'required|date',
            'transaction_ref_id'   => 'required|string',

            //'source_type' => 'nullable|in:student,guest,organization',

            // 'source_id' => [
            //     'nullable',
            //     Rule::requiredIf(fn () =>
            //         $request->source_type === 'student'
            //     ),
            // ],

            // 'source_name' => [
            //     'nullable',
            //     Rule::requiredIf(fn () =>
            //         $request->source_type !== 'student'
            //     ),
            //     'string',
            //     'max:255',
            // ],

            'transaction_reference_photo' =>
                'nullable|file|mimes:jpg,jpeg,png,webp,pdf|max:2048',
        ]);

        /* ================= FILE UPDATE ================= */
        $referencePath = $income->transaction_reference_photo;

        if ($request->hasFile('transaction_reference_photo')) {
            if ($referencePath && Storage::disk('public')->exists($referencePath)) {
                Storage::disk('public')->delete($referencePath);
            }

            $referencePath = $request->file('transaction_reference_photo')
                ->store('transactions/income', 'public');
        }

        //echo $referencePath;die;

        /* ================= UPDATE ================= */
        $income->update([
            'income_head_id'       => $request->income_head_id,
            'transaction_amount'  => $request->transaction_amount,
            'payment_method'      => $request->payment_method,
            'transaction_date'    => $request->transaction_date,
            'transaction_ref_id'  => $request->transaction_ref_id,
            'transaction_details' => $request->transaction_details,
            'transaction_reference_photo'=> $referencePath,

            // 'source_type' => $request->source_type,
            // 'source_id'   => $request->source_type === 'student'
            //                     ? $request->source_id
            //                     : null,
            // 'source_name' => $request->source_type !== 'student'
            //                     ? $request->source_name
            //                     : null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Income updated successfully'
        ]);
    }

    public function updateExpense(Request $request, $id)
    {
        $expense = TransactionExpense::findOrFail($id);

        $request->validate([
            'expense_head_id'      => 'required|exists:expense_heads,id',
            'transaction_amount'  => 'required|numeric|min:1',
            'payment_method'      => 'required|string',
            'transaction_date'    => 'required|date',
            'transaction_ref_id'  => 'required|string',

            'source_type'          => 'nullable|in:staff,vendor',

            'source_id' => [
                'nullable',
                Rule::requiredIf(fn () =>
                    $request->source_type === 'staff'
                ),
            ],

            'source_name' => [
                'nullable',
                'string',
                'max:255',
            ],

            'transaction_reference_photo' =>
                'nullable|file|mimes:jpg,jpeg,png,webp,pdf|max:2048',
        ]);

        /* ================= FILE UPDATE ================= */
        $referencePath = $expense->transaction_reference_photo;

        if ($request->hasFile('transaction_reference_photo')) {
            if ($referencePath && Storage::disk('public')->exists($referencePath)) {
                Storage::disk('public')->delete($referencePath);
            }

            $referencePath = $request->file('transaction_reference_photo')
                ->store('transactions/expense', 'public');
        }
        //echo $referencePath;die;

        /* ================= UPDATE ================= */
        $expense->update([
            'expense_head_id'      => $request->expense_head_id,
            'transaction_amount'  => $request->transaction_amount,
            'payment_method'      => $request->payment_method,
            'transaction_date'    => $request->transaction_date,
            'transaction_ref_id'  => $request->transaction_ref_id,
            'transaction_details' => $request->transaction_details,
            'transaction_reference_photo'=> $referencePath,

            'source_type' => $request->source_type,
            'source_id'   => $request->source_id ?? null,
            'source_name' => $request->source_name ?? null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Expense updated successfully'
        ]);
    }


    public function destroy(string $type, int $id)
    {
        try {

            $trans_head_id ="";
            if ($type === 'income') {
                $transaction = TransactionIncome::findOrFail($id);
                $trans_head_id = $transaction->income_head_id;
            } elseif ($type === 'expense') {
                $transaction = TransactionExpense::findOrFail($id);
                $trans_head_id = $transaction->expense_head_id;
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Invalid transaction type'
                ], 422);
            }

            // 🔹 Delete reference file if exists
            if ($transaction->transaction_ref_file) {
                \Storage::disk('public')->delete($transaction->transaction_ref_file);
            }

            if($trans_head_id != 1) {
                $transaction->delete();

                return response()->json([
                    'status' => true,
                    'message' => ucfirst($type) . ' deleted successfully'
                ]);
            }

        } catch (\Throwable $e) {

            return response()->json([
                'status' => false,
                'message' => 'Unable to delete transaction'
            ], 500);
        }
    }



}

