<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Project;
use App\Models\ProjectResourceMapping;
use App\Models\User;
use Dompdf\Dompdf;
use Dompdf\Options;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Yajra\DataTables\Contracts\DataTable;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Str;
use App\Models\ProjectBillingModule;
use Illuminate\Support\Arr;

class ProjectController extends Controller
{
    public function index()
    {
        $users = User::whereIn('role_id', [2, 3, 4])->get();
        // return $users;
        return view('auth.project.add', compact('users'));
    }

    public function submit(Request $request)
    {
        $validated = $request->validate([
            'project_name'   => 'required|string|max:255',
            'project_TL_id'  => 'required|exists:users,id',
            'project_manager_id' => 'required|exists:users,id',
            'project_type'   => 'required|integer',
            'client_name'    => 'required|string|max:255',
            // 'project_type_note' => 'required|string|max:255',
            'technology'          => 'required|array|min:1',
            'technology.*'        => 'string',
            'assigned_employee'          => 'required|array|min:1',
            'assigned_employee.*'        => 'integer',
            'start_date'     => 'required|date',
            'end_date'       => 'required|date|after_or_equal:start_date',
            'total_resource_cost' => 'required',
            'pm_cost' => 'required',
            // 'profit_margin' => 'required',
            // 'marketing_cost' => 'required',
            // 'sales_cost' => 'required',
            'total_project_cost' => 'required',
            // 'billing_type' => 'required|in:normal,module_wise,advance,postpayment',
        ]);

        $start_date = Carbon::parse($request->start_date)->format('Y-m-d');
        $end_date = Carbon::parse($request->end_date)->format('Y-m-d');

        $rawCosts = $request->input('other_costs', []);
        $finalCosts = [];

        $fixedCosts = ['accommodation_cost', 'commission', 'travelling_cost'];

        foreach ($fixedCosts as $costKey) {
            if (isset($rawCosts[$costKey]) && $rawCosts[$costKey] !== '') {
                $finalCosts[$this->formatCostLabel($costKey)] = (float) $rawCosts[$costKey];
            }
        }

        foreach ($rawCosts as $key => $value) {
            if ($key !== 'custom') {
                if (!is_null($value) && $value !== '') {
                    $finalCosts[$this->formatCostLabel($key)] = (float) $value;
                }
            }
        }
        if (isset($rawCosts['custom']) && is_array($rawCosts['custom'])) {
            foreach ($rawCosts['custom'] as $custom) {
                if (!empty($custom['title']) && isset($custom['amount'])) {
                    $finalCosts[$custom['title']] = (float) $custom['amount'];
                }
            }
        }
        // return $finalCosts;

        $project = Project::create([
            'project_name'   => $request->project_name,
            'project_TL_id'  => $request->project_TL_id,
            'project_type'   => $request->project_type,
            'project_type_note' => $request->project_type_note,
            'client_name'    => $request->client_name,
            'project_manager_id' => $request->project_manager_id,
            'technology'      => json_encode($request->technology),
            'assigned_employee' => json_encode($request->assigned_employee),
            'start_date'     => $start_date,
            'end_date'       => $end_date,
            'description'    => $request->description,
            'status'         => $request->status,
            'total_resource_cost' => $request->total_resource_cost,
            'pm_cost' => $request->pm_cost,
            'operational_cost' => $request->infra_cost,
            'travel_cost' => $request->travel_cost,
            'comission' => $request->comission,
            'accomodation_cost' => $request->accomodation_cost,
            'profit_margin' => $request->profit_margin,
            'marketing_cost' => $request->marketing_cost,
            'sales_cost' => $request->sales_cost,
            'total_project_cost' => $request->total_project_cost,
            'other_costs' => json_encode($finalCosts),
        ]);

        $projectId = $project->project_id;
        $billingType = $request->billing_type;

        foreach ($request->resources as $resourceData) {
            ProjectResourceMapping::create([
                'project_id' => $project->project_id,
                'resource_id' => $resourceData['user_id'],
                'name' => $resourceData['name'],
                'cost_per_hour' => $resourceData['rate'],
                'duration' => $resourceData['duration'],
                'total_cost' => $resourceData['total'],
            ]);
        }

        foreach ($request->modules as $module) {
            ProjectBillingModule::create([
                'project_id' => $projectId,
                'title' => $module['title'],
                'amount' => $module['amount'],
                'description' => $module['description'],
                'start_date' =>  $module['start_date'],
                'end_date' =>  $module['end_date'],
                'status' => 'pending',
            ]);
        }

        return redirect()->route('project.manage.view')->with('success', 'Project created successfully!');
    }

    private function formatCostLabel($key)
    {
        $labelMap = [
            'travel_cost' => 'Travelling Cost',
            'comission' => 'Comission',
            'accomodation_cost' => 'Accomodation Cost'
        ];

        return $labelMap[$key] ?? $key;
    }
    public function manageView()
    {
        return view('auth.project.manage');
    }
    public function getProjectData(Request $request)
    {
        $data = Project::all();
        return DataTables::of($data)
            // ->addColumn('user_name', function ($row) {
            //     return optional($row->user)->name ?? '-';
            // })
            ->editColumn('technology', function ($row) {
                $technologies = json_decode($row->technology, true); // decode to array
                $badges = '';
                if (is_array($technologies)) {
                    foreach ($technologies as $technology) {
                        $badges .= '<span class="badge bg-primary me-1">' . e($technology) . '</span>';
                    }
                }
                return $badges;
            })
            ->addColumn('action', function ($row) {
                $editUrl = route('project.edit', $row->project_id);
                $detailsUrl = route('project.details', $row->project_id);

                return '
            <a href="' . $detailsUrl . '" class="btn btn-sm btn-secondary details-btn">Details</a>
            <a href="' . $editUrl . '" class="btn btn-sm btn-warning me-1">Edit</a>
            <button data-id="' . $row->project_id . '" class="btn btn-sm btn-danger delete-btn">Delete</button>
        ';
            })
            ->rawColumns(['technology', 'action'])
            ->make(true);
    }
    public function edit($id)
    {

        $project = Project::with(['resources', 'billingModules'])->findOrFail($id);

        $users = User::whereIn('role_id', [2, 3, 4])->get();
        $resources = ProjectResourceMapping::where('project_id', $id)->get();
        // return $resources;
        $otherCosts = json_decode($project->other_costs, true);

        $billingModules = $project->billingModules;
        // return $billingModules;

        return view('auth.project.edit', compact('users', 'project', 'resources', 'otherCosts', 'billingModules'));
    }

    public function editSubmit(Request $request, $id)
    {
        // return $request;
        $project = Project::with(['resources', 'billingModules'])->findOrFail($id);
        // return $project; comming data

        $start_date = Carbon::parse($request->start_date)->format('Y-m-d');
        $end_date = Carbon::parse($request->end_date)->format('Y-m-d');

        // 1) Load existing costs CORRECTLY from the project and clean them
        $existingCosts = json_decode($project->other_costs ?? '[]', true) ?: [];
        $existingCosts = $this->flattenHistoryObjects($existingCosts); // removes changed_at etc.

        // 2) Build $finalCosts (your code stays the same)
        $rawCosts   = $request->input('other_costs', []);
        $finalCosts = [];
        $fixedCosts = ['accommodation_cost', 'commission', 'travelling_cost'];

        foreach ($fixedCosts as $costKey) {
            if (isset($rawCosts[$costKey]) && $rawCosts[$costKey] !== '') {
                $finalCosts[$this->formatCostLabel($costKey)] = (float) $rawCosts[$costKey];
            }
        }

        foreach ($rawCosts as $key => $value) {
            if ($key !== 'custom' && !in_array($key, $fixedCosts, true)) {
                if (!is_null($value) && $value !== '') {
                    $finalCosts[$this->formatCostLabel($key)] = (float) $value;
                }
            }
        }

        if (isset($rawCosts['custom']) && is_array($rawCosts['custom'])) {
            foreach ($rawCosts['custom'] as $custom) {
                if (!empty($custom['title']) && isset($custom['amount'])) {
                    $finalCosts[$custom['title']] = (float) $custom['amount'];
                }
            }
        }

        // 3) Merge WITHOUT changed_at, keeping old keys
        $merged = $this->mergeCostsNumeric($existingCosts, $finalCosts);

        // Update project
        $project->update([
            'project_name'   => $request->project_name,
            'project_TL_id'  => $request->project_TL_id,
            'project_type'   => $request->project_type,
            'project_type_note' => $request->project_type_note,
            'client_name'    => $request->client_name,
            'project_manager_id' => $request->project_manager_id,
            'technology'      => json_encode($request->technology),
            'assigned_employee' => json_encode($request->assigned_employee),
            'start_date'     => $start_date,
            'end_date'       => $end_date,
            'description'    => $request->description,
            'status'         => $request->status,
            'total_resource_cost' => $request->total_resource_cost,
            'pm_cost' => $request->pm_cost,
            'operational_cost' => $request->infra_cost,
            'travel_cost' => $request->travel_cost,
            'comission' => $request->comission,
            'accomodation_cost' => $request->accomodation_cost,
            'profit_margin' => $request->profit_margin,
            'marketing_cost' => $request->marketing_cost,
            'sales_cost' => $request->sales_cost,
            'total_project_cost' => $request->total_project_cost,
            'other_costs' => json_encode($merged),
        ]);

        // // Remove existing mappings
        // ProjectResourceMapping::where('project_id', $project->project_id)->delete();
        // ProjectBillingModule::where('project_id', $project->project_id)->delete();

        // Re-insert resource mappings
        DB::transaction(function () use ($request, $project) {
            $projectId = $project->project_id;
            $existing = ProjectResourceMapping::where('project_id', $projectId)
                ->orderBy('created_at', 'desc')
                ->get()
                ->groupBy('resource_id');

            $rowsToInsert = [];
            $now = now();

            foreach ($request->resources as $resourceData) {
                $resourceId = $resourceData['user_id'];
                $incoming = [
                    'name'          => trim($resourceData['name'] ?? ''),
                    'cost_per_hour' => (float) ($resourceData['rate'] ?? 0),
                    'duration'      => (float) ($resourceData['duration'] ?? 0),
                    'total_cost'    => (float) ($resourceData['total'] ?? 0),
                ];

                $collection = $existing->get($resourceId, collect());
                $latest = $collection->first();

                $shouldInsert = false;

                if (!$latest) {
                    $shouldInsert = true;
                } else {
                    $changed =
                        $latest->name !== $incoming['name'] ||
                        !$this->floatsEqual($latest->cost_per_hour, $incoming['cost_per_hour']) ||
                        !$this->floatsEqual($latest->duration, $incoming['duration']) ||
                        !$this->floatsEqual($latest->total_cost, $incoming['total_cost']);

                    $shouldInsert = $changed;
                }

                if ($shouldInsert) {
                    $rowsToInsert[] = [
                        'project_id'    => $projectId,
                        'resource_id'   => $resourceId,
                        'name'          => $incoming['name'],
                        'cost_per_hour' => $incoming['cost_per_hour'],
                        'duration'      => $incoming['duration'],
                        'total_cost'    => $incoming['total_cost'],

                    ];
                }
            }

            if (!empty($rowsToInsert)) {
                ProjectResourceMapping::insert($rowsToInsert);
            }
        });

        // Re-insert billing modules
        DB::transaction(function () use ($request, $project) {
            $projectId = $project->id;
            $existing = ProjectBillingModule::where('project_id', $projectId)
                ->latest('id')  // or created_at
                ->get()
                ->groupBy(function ($m) {
                    return Str::lower(trim($m->title ?? ''));
                });

            $rowsToInsert = [];
            $now = now();

            foreach ($request->modules ?? [] as $module) {
                $incoming = $this->normalizeIncomingModule($module);
                $key = Str::lower(trim($incoming['title']));

                $latest = optional($existing->get($key))->first();

                if (!$latest || !$this->modulesAreSame($latest, $incoming)) {
                    $rowsToInsert[] = [
                        'project_id'  => $projectId,
                        'title'       => $incoming['title'],
                        'amount'      => $incoming['amount'],
                        'description' => $incoming['description'],
                        'start_date'  => $incoming['start_date'],
                        'end_date'    => $incoming['end_date'],
                        'status'      => $incoming['status'] ?? 'pending',
                        'created_at'  => $now,
                        'updated_at'  => $now,
                    ];
                }
            }

            if (!empty($rowsToInsert)) {
                ProjectBillingModule::insert($rowsToInsert);
            }
        });

        return redirect()->route('project.manage.view')->with('success', 'Project updated successfully!');
    }
    /**
     * Merge rule (number / array of numbers, NO changed_at):
     * - New key -> save as number
     * - Same key, same value -> do nothing
     * - Same key, different value -> turn into array and append
     */
    private function mergeCostsNumeric(array $existing, array $incoming, int $precision = 2): array
    {
        foreach ($incoming as $label => $newAmount) {
            $newAmount = (float) $newAmount;

            if (!array_key_exists($label, $existing)) {
                $existing[$label] = $newAmount;
                continue;
            }

            $old = $existing[$label];

            // if it's an array of numbers already
            if (is_array($old) && $this->isNumericArray($old)) {
                $last = end($old);
                if (!$this->amountsAreEqual((float) $last, $newAmount, $precision)) {
                    $existing[$label][] = $newAmount;
                }
                continue;
            }

            // if it's a single number
            if (!is_array($old)) {
                $oldAmount = (float) $old;
                if (!$this->amountsAreEqual($oldAmount, $newAmount, $precision)) {
                    $existing[$label] = [$oldAmount, $newAmount];
                }
                continue;
            }

            // Defensive: if it's some other array structure, flatten to numbers
            $flatOld = $this->flattenHistoryObjects([$label => $old])[$label];
            if (is_array($flatOld)) {
                $last = end($flatOld);
                if (!$this->amountsAreEqual((float) $last, $newAmount, $precision)) {
                    $flatOld[] = $newAmount;
                }
                $existing[$label] = $flatOld;
            } else {
                if (!$this->amountsAreEqual((float) $flatOld, $newAmount, $precision)) {
                    $existing[$label] = [(float) $flatOld, $newAmount];
                }
            }
        }

        return $existing;
    }
    /**
     * Turn structures like:
     *   "Commission": [{"amount":1000,"changed_at":"..."}, {"amount":1200,"changed_at":"..."}]
     * into:
     *   "Commission": [1000, 1200]
     * or single number if only 1.
     */
    private function flattenHistoryObjects(array $data)
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                // array of objects?
                if ($this->isAssocArrayOfAmountObjects($value)) {
                    $amounts = array_values(array_map(
                        fn($row) => (float) ($row['amount'] ?? 0),
                        $value
                    ));
                    $data[$key] = count($amounts) === 1 ? $amounts[0] : $amounts;
                } elseif ($this->isNumericArray($value)) {
                    // already fine
                    $data[$key] = count($value) === 1 ? (float) $value[0] : array_map('floatval', $value);
                } else {
                    // nested weird arrays – try to recurse
                    $data[$key] = $this->flattenHistoryObjects($value);
                }
            } elseif (is_numeric($value)) {
                $data[$key] = (float) $value;
            }
        }

        return $data;
    }
    private function isAssocArrayOfAmountObjects(array $arr): bool
    {
        // array of arrays that have 'amount' key
        foreach ($arr as $v) {
            if (!is_array($v) || !array_key_exists('amount', $v)) {
                return false;
            }
        }
        return true;
    }
    private function isNumericArray(array $arr): bool
    {
        foreach ($arr as $v) {
            if (!is_numeric($v)) return false;
        }
        return true;
    }
    private function amountsAreEqual($a, $b, int $precision = 2): bool
    {
        $a = is_numeric($a) ? (float) $a : 0.0;
        $b = is_numeric($b) ? (float) $b : 0.0;

        return round($a, $precision) === round($b, $precision);
    }
    private function floatsEqual(float $a, float $b, int $precision = 2): bool
    {
        return round($a, $precision) === round($b, $precision);
    }
    private function normalizeIncomingModule(array $module): array
    {
        return [
            'title'       => trim((string) ($module['title'] ?? '')),
            'amount'      => isset($module['amount']) ? (float) $module['amount'] : 0.0,
            'description' => $module['description'] ?? null,
            'start_date'  => !empty($module['start_date']) ? Carbon::parse($module['start_date'])->toDateString() : null,
            'end_date'    => !empty($module['end_date']) ? Carbon::parse($module['end_date'])->toDateString() : null,
            'status'      => $module['status'] ?? 'pending',
        ];
    }

    /**
     * Compare latest stored row vs incoming data
     */
    private function moduleChanged(ProjectBillingModule $latest, array $incoming): bool
    {
        return
            $latest->title !== $incoming['title'] ||
            !$this->floatsEqual((float) $latest->amount, (float) $incoming['amount']) ||
            (string) ($latest->description ?? '') !== (string) ($incoming['description'] ?? '') ||
            $this->dateEquals((string) $latest->start_date, $incoming['start_date']) === false ||
            $this->dateEquals((string) $latest->end_date, $incoming['end_date']) === false ||
            (string) $latest->status !== (string) ($incoming['status'] ?? 'pending');
    }
    private function dateEquals(?string $a, ?string $b): bool
    {
        if ($a === null && $b === null) return true;
        if ($a === null || $b === null) return false;

        return Carbon::parse($a)->eq(Carbon::parse($b));
    }

    // public function editSubmit(Request $request, $id)
    // {
    //     $project = Project::findOrFail($id);
    //     $validated = $request->validate([
    //         'project_name'   => 'required|string|max:255',
    //         'project_TL_id'  => 'required|exists:users,id',
    //         'project_manager_id' => 'required|exists:users,id',
    //         'project_type'   => 'required|integer',
    //         'client_name'    => 'required|string|max:255',
    //         'project_type_note' => 'required|string|max:255',
    //         'technology'          => 'required|array|min:1',
    //         'technology.*'        => 'string',
    //         'assigned_employee'          => 'required|array|min:1',
    //         'assigned_employee.*'        => 'integer',
    //         'total_resource_cost' => 'required',
    //         'pm_cost' => 'required',
    //         'infra_cost' => 'required',
    //         'travel_cost' => 'required',
    //         'comission' => 'required',
    //         'accomodation_cost' => 'required',
    //         'profit_margin' => 'required',
    //         'marketing_cost' => 'required',
    //         'sales_cost' => 'required',
    //         'total_project_cost' => 'required',
    //         'billing_type' => 'required|in:normal,module_wise,advance,postpayment',
    //         // 'start_date'     => 'date',
    //         // 'end_date'       => 'date|after_or_equal:start_date',
    //     ]);
    //     $project->update([
    //         'project_name'   => $request->project_name ? $request->project_name : $project->project_name,
    //         'project_TL_id'  => $request->project_TL_id ? $request->project_TL_id : $project->project_TL_id,
    //         'project_type'   => $request->project_type ? $request->project_type : $project->project_type,
    //         'project_type_note' => $request->project_type_note ? $request->project_type_note : $project->project_type_note,
    //         'client_name'    => $request->client_name ? $request->client_name : $project->client_name,
    //         'project_manager_id' => $request->project_manager_id ? $request->project_manager_id : $project->project_manager_id,
    //         'technology'      => $request->technology ?  json_encode($request->technology) : $project->technology,
    //         'assigned_employee' => $request->assigned_employee ?  json_encode($request->assigned_employee) : $project->assigned_employee,
    //         'start_date'     => $request->start_date ? $request->start_date : $project->start_date,
    //         'end_date'       => $request->end_date ? $request->end_date : $project->end_date,
    //         'description'    => $request->description ? $request->description : $project->description,
    //         'status'         => $request->status ? $request->status : $project->status,

    //         'total_resource_cost' => $request->total_resource_cost ? $request->total_resource_cost : $project->total_resource_cost,
    //         'pm_cost' => $request->pm_cost ? $request->pm_cost : $project->pm_cost,
    //         'infra_cost' => $request->infra_cost ? $request->infra_cost : $project->infra_cost,
    //         'travel_cost' => $request->travel_cost ? $request->travel_cost : $project->travel_cost,
    //         'comission' => $request->comission ? $request->comission : $project->comission,
    //         'accomodation_cost' => $request->accomodation_cost ? $request->accomodation_cost : $project->accomodation_cost,
    //         'profit_margin' => $request->profit_margin ? $request->profit_margin : $project->profit_margin,
    //         'marketing_cost' => $request->marketing_cost ? $request->marketing_cost : $project->marketing_cost,
    //         'sales_cost' => $request->sales_cost ? $request->sales_cost : $project->sales_cost,
    //         'total_project_cost' => $request->total_project_cost ? $request->total_project_cost : $project->total_project_cost,
    //         'billing_type' => $request->billing_type,
    //     ]);

    //     $billingType = $request->billing_type;

    //     $project->billingModules()->delete();

    //     if ($billingType === 'normal') {
    //         $details = [
    //             [
    //                 'title' => 'Advance Payment',
    //                 'amount' => $request->advance_payment,
    //                 'description' => $request->advance_description,
    //                 'start_date' => $request->advance_start_date,
    //                 'end_date' => $request->advance_end_date
    //             ],
    //             [
    //                 'title' => 'UAT Payment',
    //                 'amount' => $request->uat_payment,
    //                 'description' => $request->uat_description,
    //                 'start_date' => $request->uat_start_date,
    //                 'end_date' => $request->uat_end_date
    //             ],
    //             [
    //                 'title' => 'Go-Live Payment',
    //                 'amount' => $request->golive_payment,
    //                 'description' => $request->golive_description,
    //                 'start_date' => $request->golive_start_date,
    //                 'end_date' => $request->golive_end_date
    //             ],
    //         ];

    //         foreach ($details as $detail) {
    //             ProjectBillingModule::updateOrCreate(
    //                 ['project_id' => $project->project_id, 'title' => $detail['title']],
    //                 [
    //                     'amount' => $detail['amount'],
    //                     'description' => $detail['description'],
    //                     'start_date' => $detail['start_date'],
    //                     'end_date' => $detail['end_date'],
    //                     'status' => 'pending',
    //                 ]
    //             );
    //         }
    //     } elseif ($billingType === 'module_wise') {
    //         foreach ($request->modules as $index => $module) {
    //             ProjectBillingModule::updateOrCreate(
    //                 ['project_id' => $project->project_id, 'title' => $module['title']],
    //                 [
    //                     'amount' => $module['amount'],
    //                     'description' => $module['description'],
    //                     'start_date' => $module['start_date'],
    //                     'end_date' => $module['end_date'],
    //                     'status' => 'pending',
    //                 ]
    //             );
    //         }
    //     } elseif (in_array($billingType, ['advance', 'postpayment'])) {
    //         $title = ucfirst($billingType) . ' Billing';
    //         ProjectBillingModule::updateOrCreate(
    //             ['project_id' => $project->project_id, 'title' => $title],
    //             [
    //                 'amount' => $request->billing_note,
    //                 'description' => $request->billing_description,
    //                 'start_date' => $request->billing_start_date,
    //                 'end_date' => $request->billing_end_date,
    //                 'status' => 'pending',
    //             ]
    //         );
    //     }
    //     return redirect()->route('project.manage.view')->with('success', 'Project updated successfully!');
    // }
    public function destroy($project_id)
    {
        $project = Project::findOrFail($project_id);
        $project->delete();
        return redirect()->back()->with('error', 'Project deleted successfully.');
    }
    public function detailsView($id)
    {
        $project = Project::where('project_id', $id)->first();
        // $list = json_decode($project->assigned_employee);
        $users = User::all();
        return view('auth.project.detail', compact('project', 'users'));
    }
    public function downloadDetails($id)
    {
        $project = Project::findOrFail($id);
        $users = User::all();

        // to 
        $options = new Options();
        $options->set('isRemoteEnabled', true);
        $options->set('isHtml5ParserEnabled', true);
        $options->set('defaultFont', 'Arial');
        $dompdf = new Dompdf($options);

        $html = View::make('auth.project.pdf', compact('project', 'users'))->render();
        // $dompdf = new Dompdf();

        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return response($dompdf->output(), 200)
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'inline; filename="project-details.pdf"');
    }
    public function billingView()
    {
        $projects = Project::all();
        return view('auth.project.billing', compact('projects'));
    }
    public function loadBillingModules($id)
    {
        $billingProject = Project::with('billingModules')->findOrFail($id);
        $projects = Project::all();
        return response()->json([
            'project' => $billingProject,
            'billingModules' => $billingProject->billingModules,
        ]);
    }
    public function changeStatus($projectid, $moduleid)
    {
        $module = ProjectBillingModule::where('id', $moduleid)
            ->where('project_id', $projectid)
            ->firstOrFail();

        $module->status = 'paid';
        $module->save();

        return response()->json(['success' => true]);
    }
}
