<?php

namespace Drupal\Tests\api_module\Unit;

use Drupal\api_module\Controller\HomeTabServiceController;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\file\FileInterface;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;

/**
 * Unit tests for HomeTabServiceController::getTabContent.
 *
 * @group api_module
 */
class HomeTabServiceControllerTest extends UnitTestCase
{

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    // Create a container that Drupal accepts
    $container = new \Symfony\Component\DependencyInjection\ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Test case: Missing language code parameter.
   */
  public function testMissingLanguageCode()
  {
    $controller = new HomeTabServiceController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter.
   */
  public function testEmptyLanguageCode()
  {
    $controller = new HomeTabServiceController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with valid language code and tab content data.
   */
  public function testSuccessFlowWithTabContent()
  {
    // Mock database result rows
    $row1 = (object) [
      'nid' => '1',
      'langcode' => 'en',
      'tab_name' => 'Tab 1',
      'tab_image' => '10',
      'tab_feature_image' => '11',
      'icon' => '12',
      'teaser_description' => 'Teaser description 1',
      'detailed_description' => 'Detailed description 1',
      'feature_list' => 'Feature 1',
      'feature_tab_url' => 'https://example.com/tab1',
    ];

    $row2 = (object) [
      'nid' => '1',
      'langcode' => 'en',
      'tab_name' => 'Tab 1',
      'tab_image' => '10',
      'tab_feature_image' => NULL,
      'icon' => '12',
      'teaser_description' => 'Teaser description 1',
      'detailed_description' => 'Detailed description 1',
      'feature_list' => 'Feature 2',
      'feature_tab_url' => 'https://example.com/tab1',
    ];

    $row3 = (object) [
      'nid' => '2',
      'langcode' => 'en',
      'tab_name' => 'Tab 2',
      'tab_image' => '20',
      'tab_feature_image' => '21',
      'icon' => '22',
      'teaser_description' => 'Teaser description 2',
      'detailed_description' => 'Detailed description 2',
      'feature_list' => NULL,
      'feature_tab_url' => 'https://example.com/tab2',
    ];

    // Create an iterable statement mock
    // StatementInterface extends \Traversable, so we need to make it iterable
    $statement = new class($row1, $row2, $row3) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1, $row2, $row3) {
        $this->rows = [$row1, $row2, $row3];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock file entities
    $file1 = $this->createMock(FileInterface::class);
    $file1->method('getFileUri')->willReturn('public://tab_image1.jpg');

    $file2 = $this->createMock(FileInterface::class);
    $file2->method('getFileUri')->willReturn('public://tab_feature_image1.jpg');

    $file3 = $this->createMock(FileInterface::class);
    $file3->method('getFileUri')->willReturn('public://icon1.png');

    $file4 = $this->createMock(FileInterface::class);
    $file4->method('getFileUri')->willReturn('public://tab_image2.jpg');

    $file5 = $this->createMock(FileInterface::class);
    $file5->method('getFileUri')->willReturn('public://tab_feature_image2.jpg');

    $file6 = $this->createMock(FileInterface::class);
    $file6->method('getFileUri')->willReturn('public://icon2.png');

    // Mock File::load to return appropriate files
    $fileStorage = $this->getMockBuilder('Drupal\file\Entity\File')
      ->disableOriginalConstructor()
      ->onlyMethods(['load'])
      ->getMock();

    // Use static method mocking via container
    $fileStorage->expects($this->any())
      ->method('load')
      ->willReturnCallback(function ($fid) use ($file1, $file2, $file3, $file4, $file5, $file6) {
        $fileMap = [
          '10' => $file1,
          '11' => $file2,
          '12' => $file3,
          '20' => $file4,
          '21' => $file5,
          '22' => $file6,
        ];
        return $fileMap[$fid] ?? NULL;
      });

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Mock file storage to handle File::load() calls
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')
      ->willReturnCallback(function ($fid) use ($file1, $file2, $file3, $file4, $file5, $file6) {
        $fileMap = [
          '10' => $file1,
          '11' => $file2,
          '12' => $file3,
          '20' => $file4,
          '21' => $file5,
          '22' => $file6,
        ];
        return $fileMap[$fid] ?? NULL;
      });

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository (needed for File::load() static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('file_url_generator', $fileUrlGenerator);
    $container->set('language_manager', $languageManager);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-home-tab-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeTabServiceController();
    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About Content data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);

    // Check first tab data
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
    $this->assertEquals('Tab 1', $data['data'][0]['tab_name']);
    $this->assertNotEmpty($data['data'][0]['tab_image']);
    $this->assertNotEmpty($data['data'][0]['icon']);
    $this->assertEquals('Teaser description 1', $data['data'][0]['teaser_description']);
    $this->assertEquals('Detailed description 1', $data['data'][0]['detailed_description']);
    $this->assertIsArray($data['data'][0]['feature_list']);
    $this->assertCount(1, $data['data'][0]['feature_list']);
    $this->assertEquals('Feature 2', $data['data'][0]['feature_list'][0]);

    // Check second tab data
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('Tab 2', $data['data'][1]['tab_name']);
    $this->assertIsArray($data['data'][1]['feature_list']);
    $this->assertEmpty($data['data'][1]['feature_list']);
  }

  /**
   * Test case: Success flow with no tab content found.
   */
  public function testSuccessFlowWithNoTabContent()
  {
    // Create an empty iterable statement mock
    $statement = new class() implements StatementInterface, \Iterator {
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Mock entity type manager (needed for File::load)
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository (needed for File::load() static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-home-tab-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeTabServiceController();
    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About Content data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Tab content with empty image fields.
   */
  public function testTabContentWithEmptyImageFields()
  {
    // Mock database result row with NULL image fields
    $row = (object) [
      'nid' => '1',
      'langcode' => 'en',
      'tab_name' => 'Tab 1',
      'tab_image' => NULL,
      'tab_feature_image' => NULL,
      'icon' => NULL,
      'teaser_description' => 'Teaser description',
      'detailed_description' => 'Detailed description',
      'feature_list' => NULL,
      'feature_tab_url' => 'https://example.com/tab1',
    ];

    // Create an iterable statement mock with one row
    $statement = new class($row) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row) {
        $this->rows = [$row];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Mock entity type manager (needed for File::load)
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository (needed for File::load() static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-home-tab-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeTabServiceController();
    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('', $data['data'][0]['tab_image']);
    $this->assertEquals('', $data['data'][0]['icon']);
    $this->assertEquals('', $data['data'][0]['feature_image']);
    $this->assertIsArray($data['data'][0]['feature_list']);
    $this->assertEmpty($data['data'][0]['feature_list']);
  }

  /**
   * Test case: Invalid language code (language not found).
   */
  public function testInvalidLanguageCode()
  {
    // Create an empty iterable statement mock
    $statement = new class() implements StatementInterface, \Iterator {
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager - return NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Mock entity type manager (needed for File::load)
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository (needed for File::load() static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-home-tab-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new HomeTabServiceController();
    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Exception handling.
   */
  public function testExceptionHandling()
  {
    // Mock database connection that throws exception
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->willThrowException(new \Exception('Database connection failed'));

    // Mock entity type manager (needed for File::load)
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository (needed for File::load() static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-home-tab-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeTabServiceController();
    $response = $controller->getTabContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
  }

}

