<?php

namespace Drupal\Tests\api_module\Unit;

use Drupal\api_module\Controller\HomeContentController;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\file\FileInterface;
use Drupal\node\Entity\Node;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;

/**
 * Unit tests for HomeContentController.
 *
 * @group api_module
 */
class HomeContentControllerTest extends UnitTestCase
{

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    // Create a container that Drupal accepts
    $container = new \Symfony\Component\DependencyInjection\ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Test case: Missing language code parameter.
   */
  public function testGetAwardContentMissingLanguageCode()
  {
    $controller = new HomeContentController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter.
   */
  public function testGetAwardContentEmptyLanguageCode()
  {
    $controller = new HomeContentController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with award content data.
   */
  public function testGetAwardContentSuccessFlow()
  {
    // Mock node IDs
    $nids = ['1', '2'];

    // Mock entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn($nids);

    // Mock node entities
    $node1 = $this->createMock(Node::class);
    $node1->method('id')->willReturn('1');
    $node1->method('hasTranslation')->with('en')->willReturn(TRUE);
    $node1->method('getTranslation')->with('en')->willReturnSelf();

    $node2 = $this->createMock(Node::class);
    $node2->method('id')->willReturn('2');
    $node2->method('hasTranslation')->with('en')->willReturn(FALSE);

    // Mock file entities
    $file1 = $this->createMock(FileInterface::class);
    $file1->method('getFileUri')->willReturn('public://award_image1.jpg');

    $file2 = $this->createMock(FileInterface::class);
    $file2->method('getFileUri')->willReturn('public://award_image2.jpg');

    // Mock field for node1 - entity is accessed as a property via __get on the first field item
    $field1 = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $field1->method('isEmpty')->willReturn(FALSE);
    $fieldItem1 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $fieldItem1->method('__get')->willReturnCallback(function ($name) use ($file1) {
      return $name === 'entity' ? $file1 : NULL;
    });
    $field1->method('first')->willReturn($fieldItem1);
    $field1->method('__get')->willReturnCallback(function ($name) use ($fieldItem1) {
      return $fieldItem1->__get($name);
    });

    // Mock field for node2
    $field2 = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $field2->method('isEmpty')->willReturn(FALSE);
    $fieldItem2 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $fieldItem2->method('__get')->willReturnCallback(function ($name) use ($file2) {
      return $name === 'entity' ? $file2 : NULL;
    });
    $field2->method('first')->willReturn($fieldItem2);
    $field2->method('__get')->willReturnCallback(function ($name) use ($fieldItem2) {
      return $fieldItem2->__get($name);
    });

    $node1->method('get')
      ->with('field_award_image')
      ->willReturn($field1);

    $node2->method('get')
      ->with('field_award_image')
      ->willReturn($field2);

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')
      ->willReturn($query);
    $nodeStorage->method('loadMultiple')
      ->with($nids)
      ->willReturn(['1' => $node1, '2' => $node2]);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('node');

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('file_url_generator', $fileUrlGenerator);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-award-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Award Section data fetched successfully.', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);

    // Check first award data
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
    $this->assertEquals('https://example.com/award_image1.jpg', $data['data'][0]['award_image']);

    // Check second award data
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('en', $data['data'][1]['langcode']);
    $this->assertEquals('English', $data['data'][1]['language_name']);
    $this->assertEquals('https://example.com/award_image2.jpg', $data['data'][1]['award_image']);
  }

  /**
   * Test case: Success flow with no award content found.
   */
  public function testGetAwardContentNoContentFound()
  {
    // Mock empty node IDs
    $nids = [];

    // Mock entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn($nids);

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')
      ->willReturn($query);
    $nodeStorage->method('loadMultiple')
      ->with($nids)
      ->willReturn([]);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('node');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-award-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Award Section data fetched successfully.', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Award content with empty image fields.
   */
  public function testGetAwardContentEmptyImageFields()
  {
    // Mock node IDs
    $nids = ['1'];

    // Mock entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn($nids);

    // Mock node entity
    $node1 = $this->createMock(Node::class);
    $node1->method('id')->willReturn('1');
    $node1->method('hasTranslation')->with('en')->willReturn(FALSE);

    // Mock empty field
    $field1 = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $field1->method('isEmpty')->willReturn(TRUE);

    $node1->method('get')
      ->with('field_award_image')
      ->willReturn($field1);

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')
      ->willReturn($query);
    $nodeStorage->method('loadMultiple')
      ->with($nids)
      ->willReturn(['1' => $node1]);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('node');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-award-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('', $data['data'][0]['award_image']);
  }

  /**
   * Test case: Invalid language code (language not found).
   */
  public function testGetAwardContentInvalidLanguageCode()
  {
    // Mock node IDs
    $nids = ['1'];

    // Mock entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn($nids);

    // Mock node entity
    $node1 = $this->createMock(Node::class);
    $node1->method('id')->willReturn('1');
    $node1->method('hasTranslation')->with('invalid')->willReturn(FALSE);

    // Mock empty field
    $field1 = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $field1->method('isEmpty')->willReturn(TRUE);

    $node1->method('get')
      ->with('field_award_image')
      ->willReturn($field1);

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')
      ->willReturn($query);
    $nodeStorage->method('loadMultiple')
      ->with($nids)
      ->willReturn(['1' => $node1]);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('node');

    // Mock language manager - return NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-award-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('invalid', $data['data'][0]['language_name']);
  }

  /**
   * Test case: Exception handling.
   */
  public function testGetAwardContentExceptionHandling()
  {
    // Mock node storage that throws exception
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')
      ->willThrowException(new \Exception('Database connection failed'));

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-award-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getAwardContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
  }

  /**
   * Test case: Missing language code parameter for getWhoWeareNumberContent.
   */
  public function testGetWhoWeareNumberContentMissingLanguageCode()
  {
    $controller = new HomeContentController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter for getWhoWeareNumberContent.
   */
  public function testGetWhoWeareNumberContentEmptyLanguageCode()
  {
    $controller = new HomeContentController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with who we are number content data.
   */
  public function testGetWhoWeareNumberContentSuccessFlow()
  {
    // Mock database result rows
    $row1 = (object) [
      'nid' => '1',
      'counter_level' => 'Years of Experience',
      'counter_val' => '25',
    ];

    $row2 = (object) [
      'nid' => '2',
      'counter_level' => 'Happy Clients',
      'counter_val' => '1000',
    ];

    // Create an iterable statement mock
    $statement = new class($row1, $row2) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1, $row2) {
        $this->rows = [$row1, $row2];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-who-we-are-number-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Home who we are counter value fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);

    // Check first counter data
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
    $this->assertEquals('Years of Experience', $data['data'][0]['counter_lebel']);
    $this->assertEquals('25', $data['data'][0]['counter_value']);

    // Check second counter data
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('en', $data['data'][1]['langcode']);
    $this->assertEquals('English', $data['data'][1]['language_name']);
    $this->assertEquals('Happy Clients', $data['data'][1]['counter_lebel']);
    $this->assertEquals('1000', $data['data'][1]['counter_value']);
  }

  /**
   * Test case: Success flow with no who we are number content found.
   */
  public function testGetWhoWeareNumberContentNoContentFound()
  {
    // Create an empty iterable statement mock
    $statement = new class() implements StatementInterface, \Iterator {
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-who-we-are-number-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Home who we are counter value fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Who we are number content with NULL counter values.
   */
  public function testGetWhoWeareNumberContentWithNullValues()
  {
    // Mock database result row with NULL values
    $row = (object) [
      'nid' => '1',
      'counter_level' => NULL,
      'counter_val' => NULL,
    ];

    // Create an iterable statement mock with one row
    $statement = new class($row) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row) {
        $this->rows = [$row];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-who-we-are-number-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertNull($data['data'][0]['counter_lebel']);
    $this->assertNull($data['data'][0]['counter_value']);
  }

  /**
   * Test case: Invalid language code (language not found).
   */
  public function testGetWhoWeareNumberContentInvalidLanguageCode()
  {
    // Create an empty iterable statement mock
    $statement = new class() implements StatementInterface, \Iterator {
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager - return NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-who-we-are-number-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Exception handling for getWhoWeareNumberContent.
   */
  public function testGetWhoWeareNumberContentExceptionHandling()
  {
    // Mock database connection that throws exception
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->willThrowException(new \Exception('Database connection failed'));

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-who-we-are-number-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getWhoWeareNumberContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
  }

  /**
   * Test case: Missing language code parameter for getXeccedContentData.
   */
  public function testGetXeccedContentDataMissingLanguageCode()
  {
    $controller = new HomeContentController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter for getXeccedContentData.
   */
  public function testGetXeccedContentDataEmptyLanguageCode()
  {
    $controller = new HomeContentController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with xecced content data.
   */
  public function testGetXeccedContentDataSuccessFlow()
  {
    // Mock database result rows
    $row1 = (object) [
      'nid' => '1',
      'xecced_value' => 'Value 1',
      'icon_url' => '10',
    ];

    $row2 = (object) [
      'nid' => '2',
      'xecced_value' => 'Value 2',
      'icon_url' => '20',
    ];

    // Create an iterable statement mock
    $statement = new class($row1, $row2) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1, $row2) {
        $this->rows = [$row1, $row2];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock file entities
    $file1 = $this->createMock(FileInterface::class);
    $file1->method('getFileUri')->willReturn('public://icon1.png');

    $file2 = $this->createMock(FileInterface::class);
    $file2->method('getFileUri')->willReturn('public://icon2.png');

    // Mock file storage
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')
      ->willReturnCallback(function ($fid) use ($file1, $file2) {
        $fileMap = [
          '10' => $file1,
          '20' => $file2,
        ];
        return $fileMap[$fid] ?? NULL;
      });

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('file_url_generator', $fileUrlGenerator);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-xecced-content-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Xecced value fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);

    // Check first xecced data
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
    $this->assertEquals('Value 1', $data['data'][0]['xecced_value']);
    $this->assertEquals('https://example.com/icon1.png', $data['data'][0]['icon']);

    // Check second xecced data
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('en', $data['data'][1]['langcode']);
    $this->assertEquals('English', $data['data'][1]['language_name']);
    $this->assertEquals('Value 2', $data['data'][1]['xecced_value']);
    $this->assertEquals('https://example.com/icon2.png', $data['data'][1]['icon']);
  }

  /**
   * Test case: Success flow with no xecced content found.
   */
  public function testGetXeccedContentDataNoContentFound()
  {
    // Create an empty iterable statement mock
    $statement = new class() implements StatementInterface, \Iterator {
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Mock file storage
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-xecced-content-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Xecced value fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Xecced content with empty/null icon fields.
   */
  public function testGetXeccedContentDataWithEmptyIconFields()
  {
    // Mock database result row with NULL icon_url
    $row1 = (object) [
      'nid' => '1',
      'xecced_value' => 'Value 1',
      'icon_url' => NULL,
    ];

    $row2 = (object) [
      'nid' => '2',
      'xecced_value' => 'Value 2',
      'icon_url' => '',
    ];

    // Create an iterable statement mock
    $statement = new class($row1, $row2) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1, $row2) {
        $this->rows = [$row1, $row2];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock file storage
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-xecced-content-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('', $data['data'][0]['icon']);
    $this->assertEquals('Value 1', $data['data'][0]['xecced_value']);
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('', $data['data'][1]['icon']);
    $this->assertEquals('Value 2', $data['data'][1]['xecced_value']);
  }

  /**
   * Test case: Invalid language code (language not found).
   */
  public function testGetXeccedContentDataInvalidLanguageCode()
  {
    // Mock database result row
    $row = (object) [
      'nid' => '1',
      'xecced_value' => 'Value 1',
      'icon_url' => NULL,
    ];

    // Create an iterable statement mock
    $statement = new class($row) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row) {
        $this->rows = [$row];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('distinct')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager - return NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Mock file storage
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-xecced-content-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('invalid', $data['data'][0]['language_name']);
  }

  /**
   * Test case: Exception handling for getXeccedContentData.
   */
  public function testGetXeccedContentDataExceptionHandling()
  {
    // Mock database connection that throws exception
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->willThrowException(new \Exception('Database connection failed'));

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-xecced-content-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new HomeContentController();
    $response = $controller->getXeccedContentData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
  }

}

