<?php

namespace Drupal\Tests\api_module\Unit;

use Drupal\api_module\Controller\BannerController;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\node\NodeInterface;
use Drupal\node\Entity\Node;
use Drupal\file\FileInterface;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\File\FileUrlGeneratorInterface;

/**
 * Unit tests for BannerController::getBannerData.
 *
 * @group api_module
 */
class BannerControllerTest extends UnitTestCase
{

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    // Create a container that Drupal accepts
    $container = new \Symfony\Component\DependencyInjection\ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Test case: Missing language code parameter.
   */
  public function testMissingLanguageCode()
  {
    $controller = new BannerController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->getBannerData($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter.
   */
  public function testEmptyLanguageCode()
  {
    $controller = new BannerController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->getBannerData($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with valid language code and nodes.
   */
  public function testSuccessFlowWithNodes()
  {
    // Mock node IDs
    $nids = ['1', '2'];

    // Mock entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn($nids);

    // Mock nodes
    $node1 = $this->createMock(NodeInterface::class);
    $node1->method('id')->willReturn('1');
    $node1->method('label')->willReturn('Banner 1');
    $node1->method('hasTranslation')->willReturn(TRUE);
    $node1->method('getTranslation')->willReturnSelf();
    $node1->method('get')->willReturnCallback(function ($field) {
      $fieldItemList = $this->createMock(\Drupal\Core\Field\FieldItemListInterface::class);

      if ($field === 'field_banner_main_image' || $field === 'field_banner_logo') {
        $fieldItemList->method('isEmpty')->willReturn(FALSE);
        $file = $this->createMockFile();
        // entity is accessed as a property via __get on the first field item
        $fieldItem = $this->createMock(\Drupal\Core\Field\FieldItemInterface::class);
        $fieldItem->method('__get')->willReturnCallback(function ($name) use ($file) {
          return $name === 'entity' ? $file : null;
        });
        $fieldItemList->method('first')->willReturn($fieldItem);
        $fieldItemList->method('__get')->willReturnCallback(function ($name) use ($fieldItem) {
          return $fieldItem->__get($name);
        });
      } else {
        $fieldItemList->method('isEmpty')->willReturn(FALSE);
        $value = '';
        if ($field === 'field_banner_short_text') {
          $value = 'Banner Short Text 1';
        } elseif ($field === 'field_banner_sub_short_text') {
          $value = 'Banner Sub Short Text 1';
        } elseif ($field === 'field_banner_title') {
          $value = 'Banner Title 1';
        } elseif ($field === 'field_banner_url') {
          $value = 'https://example.com/banner1';
        } elseif ($field === 'field_banner_sequence') {
          $value = '1';
        }
        // Mock the first field item that has the value property
        $fieldItem = $this->createMock(\Drupal\Core\Field\FieldItemInterface::class);
        $fieldItem->method('__get')->willReturnCallback(function ($name) use ($value) {
          return $name === 'value' ? $value : null;
        });
        // FieldItemList delegates __get to first item via first() method
        $fieldItemList->method('first')->willReturn($fieldItem);
        $fieldItemList->method('__get')->willReturnCallback(function ($name) use ($fieldItem) {
          return $fieldItem->__get($name);
        });
      }
      return $fieldItemList;
    });

    $node2 = $this->createMock(NodeInterface::class);
    $node2->method('id')->willReturn('2');
    $node2->method('label')->willReturn('Banner 2');
    $node2->method('hasTranslation')->willReturn(FALSE);
    $node2->method('get')->willReturnCallback(function ($field) {
      $fieldItemList = $this->createMock(\Drupal\Core\Field\FieldItemListInterface::class);

      if ($field === 'field_banner_main_image' || $field === 'field_banner_logo') {
        $fieldItemList->method('isEmpty')->willReturn(FALSE);
        $file = $this->createMockFile();
        // entity is accessed as a property via __get on the first field item
        $fieldItem = $this->createMock(\Drupal\Core\Field\FieldItemInterface::class);
        $fieldItem->method('__get')->willReturnCallback(function ($name) use ($file) {
          return $name === 'entity' ? $file : null;
        });
        $fieldItemList->method('first')->willReturn($fieldItem);
        $fieldItemList->method('__get')->willReturnCallback(function ($name) use ($fieldItem) {
          return $fieldItem->__get($name);
        });
      } else {
        $fieldItemList->method('isEmpty')->willReturn(FALSE);
        $value = '';
        if ($field === 'field_banner_short_text') {
          $value = 'Banner Short Text 2';
        } elseif ($field === 'field_banner_sub_short_text') {
          $value = 'Banner Sub Short Text 2';
        } elseif ($field === 'field_banner_title') {
          $value = 'Banner Title 2';
        } elseif ($field === 'field_banner_url') {
          $value = 'https://example.com/banner2';
        } elseif ($field === 'field_banner_sequence') {
          $value = '2';
        }
        // Mock the first field item that has the value property
        $fieldItem = $this->createMock(\Drupal\Core\Field\FieldItemInterface::class);
        $fieldItem->method('__get')->willReturnCallback(function ($name) use ($value) {
          return $name === 'value' ? $value : null;
        });
        // FieldItemList delegates __get to first item via first() method
        $fieldItemList->method('first')->willReturn($fieldItem);
        $fieldItemList->method('__get')->willReturnCallback(function ($name) use ($fieldItem) {
          return $fieldItem->__get($name);
        });
      }
      return $fieldItemList;
    });

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('loadMultiple')
      ->with($nids)
      ->willReturn(['1' => $node1, '2' => $node2]);
    $nodeStorage->method('getQuery')
      ->with('AND')
      ->willReturn($query);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository (needed for Node::loadMultiple static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->with(Node::class)
      ->willReturn('node');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturn('https://example.com/files/banner.jpg');

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);
    $container->set('file_url_generator', $fileUrlGenerator);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-banner-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new BannerController();
    $response = $controller->getBannerData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Debug: Output error message if test fails
    if ($response->getStatusCode() !== 200) {
      $errorMsg = isset($data['message']) ? $data['message'] : 'Unknown error';
      $this->fail('Test failed with status ' . $response->getStatusCode() . ': ' . $errorMsg . ' | Full response: ' . $response->getContent());
    }

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Banner data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
  }

  /**
   * Test case: Success flow with no nodes found.
   */
  public function testSuccessFlowWithNoNodes()
  {
    // Mock empty entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn([]);

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('loadMultiple')
      ->with([])
      ->willReturn([]);
    $nodeStorage->method('getQuery')
      ->with('AND')
      ->willReturn($query);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository (needed for Node::loadMultiple static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->with(Node::class)
      ->willReturn('node');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-banner-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new BannerController();
    $response = $controller->getBannerData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('Banner data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Invalid language code (language not found).
   */
  public function testInvalidLanguageCode()
  {
    // Mock empty entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn([]);

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('loadMultiple')
      ->with([])
      ->willReturn([]);
    $nodeStorage->method('getQuery')
      ->with('AND')
      ->willReturn($query);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository (needed for Node::loadMultiple static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->with(Node::class)
      ->willReturn('node');

    // Mock language manager - return NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-banner-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new BannerController();
    $response = $controller->getBannerData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Debug: Output error message if test fails
    if ($response->getStatusCode() !== 200) {
      $errorMsg = isset($data['message']) ? $data['message'] : 'Unknown error';
      $this->fail('Test failed with status ' . $response->getStatusCode() . ': ' . $errorMsg . ' | Full response: ' . $response->getContent());
    }

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    // When language is invalid, it should still return success but with the langcode as language_name
    $this->assertIsArray($data['data']);
    // Since there are no nodes, data should be empty
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Node with empty image fields.
   */
  public function testNodeWithEmptyImageFields()
  {
    // Mock node IDs
    $nids = ['1'];

    // Mock entity query
    $query = $this->createMock(QueryInterface::class);
    $query->method('condition')->willReturnSelf();
    $query->method('accessCheck')->willReturnSelf();
    $query->method('execute')->willReturn($nids);

    // Mock node with empty image fields
    $node = $this->createMock(NodeInterface::class);
    $node->method('id')->willReturn('1');
    $node->method('label')->willReturn('Banner 1');
    $node->method('hasTranslation')->willReturn(FALSE);
    $node->method('get')->willReturnCallback(function ($field) {
      $fieldItemList = $this->createMock(\Drupal\Core\Field\FieldItemListInterface::class);

      if ($field === 'field_banner_main_image' || $field === 'field_banner_logo') {
        $fieldItemList->method('isEmpty')->willReturn(TRUE);
      } else {
        $fieldItemList->method('isEmpty')->willReturn(FALSE);
        $value = '';
        if ($field === 'field_banner_short_text') {
          $value = 'Banner Short Text';
        } elseif ($field === 'field_banner_sub_short_text') {
          $value = 'Banner Sub Short Text';
        } elseif ($field === 'field_banner_title') {
          $value = 'Banner Title';
        } elseif ($field === 'field_banner_url') {
          $value = 'https://example.com/banner';
        } elseif ($field === 'field_banner_sequence') {
          $value = '1';
        }
        // Mock the first field item that has the value property
        $fieldItem = $this->createMock(\Drupal\Core\Field\FieldItemInterface::class);
        $fieldItem->method('__get')->willReturnCallback(function ($name) use ($value) {
          return $name === 'value' ? $value : null;
        });
        // FieldItemList delegates __get to first item via first() method
        $fieldItemList->method('first')->willReturn($fieldItem);
        $fieldItemList->method('__get')->willReturnCallback(function ($name) use ($fieldItem) {
          return $fieldItem->__get($name);
        });
      }
      return $fieldItemList;
    });

    // Mock node storage
    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('loadMultiple')
      ->with($nids)
      ->willReturn(['1' => $node]);
    $nodeStorage->method('getQuery')
      ->with('AND')
      ->willReturn($query);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('node')
      ->willReturn($nodeStorage);

    // Mock entity type repository (needed for Node::loadMultiple static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->with(Node::class)
      ->willReturn('node');

    // Mock language manager
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-banner-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new BannerController();
    $response = $controller->getBannerData($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('', $data['data'][0]['banner_image']);
    $this->assertEquals('', $data['data'][0]['banner_logo']);
  }

  /**
   * Helper method to create a mock file.
   */
  protected function createMockFile()
  {
    $file = $this->createMock(FileInterface::class);
    $file->method('getFileUri')->willReturn('public://banner.jpg');
    return $file;
  }

}
