<?php

namespace Drupal\Tests\api_module\Unit;

use Drupal\api_module\Controller\about\AboutController;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\file\FileInterface;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;

/**
 * Unit tests for AboutController.
 *
 * @group api_module
 */
class AboutControllerTest extends UnitTestCase
{

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    // Create a container that Drupal accepts
    $container = new \Symfony\Component\DependencyInjection\ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Test case: Missing language code parameter.
   */
  public function testFaqcontentMissingLanguageCode()
  {
    $controller = new AboutController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter.
   */
  public function testFaqcontentEmptyLanguageCode()
  {
    $controller = new AboutController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with FAQ content data.
   */
  public function testFaqcontentSuccessFlow()
  {
    // Mock database result rows
    $row1 = (object) [
      'nid' => '1',
      'question' => 'What is Drupal?',
      'answer' => 'Drupal is a content management system.',
    ];
    $row2 = (object) [
      'nid' => '2',
      'question' => 'How to install Drupal?',
      'answer' => 'Follow the installation guide.',
    ];

    // Create an iterable statement mock
    $statement = new class($row1, $row2) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1, $row2) {
        $this->rows = [$row1, $row2];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('join')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-faq-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('FAQ fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);

    // Check first FAQ data
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
    $this->assertEquals('What is Drupal?', $data['data'][0]['question']);
    $this->assertEquals('Drupal is a content management system.', $data['data'][0]['answer']);

    // Check second FAQ data
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('en', $data['data'][1]['langcode']);
    $this->assertEquals('English', $data['data'][1]['language_name']);
    $this->assertEquals('How to install Drupal?', $data['data'][1]['question']);
    $this->assertEquals('Follow the installation guide.', $data['data'][1]['answer']);
  }

  /**
   * Test case: Success flow with no FAQ content found.
   */
  public function testFaqcontentNoContentFound()
  {
    // Create an iterable statement mock with no rows
    $statement = new class implements StatementInterface, \Iterator {
      private $rows = [];
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('join')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-faq-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('FAQ fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Language manager returns NULL for invalid language.
   */
  public function testFaqcontentInvalidLanguage()
  {
    // Mock database result rows
    $row1 = (object) [
      'nid' => '1',
      'question' => 'Test Question',
      'answer' => 'Test Answer',
    ];

    // Create an iterable statement mock
    $statement = new class($row1) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1) {
        $this->rows = [$row1];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('join')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager returning NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-faq-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('invalid', $data['data'][0]['language_name']);
  }

  /**
   * Test case: Exception handling.
   */
  public function testFaqcontentExceptionHandling()
  {
    // Mock database connection that throws exception
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->willThrowException(new \Exception('Database connection failed'));

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-faq-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
    $this->assertStringContainsString('Database connection failed', $data['message']);
  }

  /**
   * Test case: Success flow with single FAQ item.
   */
  public function testFaqcontentSingleItem()
  {
    // Mock database result row
    $row1 = (object) [
      'nid' => '5',
      'question' => 'What is PHP?',
      'answer' => 'PHP is a server-side scripting language.',
    ];

    // Create an iterable statement mock
    $statement = new class($row1) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1) {
        $this->rows = [$row1];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('join')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('French');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('fr')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-faq-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'fr'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->faqcontent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('FAQ fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('5', $data['data'][0]['nid']);
    $this->assertEquals('fr', $data['data'][0]['langcode']);
    $this->assertEquals('French', $data['data'][0]['language_name']);
    $this->assertEquals('What is PHP?', $data['data'][0]['question']);
    $this->assertEquals('PHP is a server-side scripting language.', $data['data'][0]['answer']);
  }

  /**
   * Test case: Missing language code parameter for aboutContent.
   */
  public function testAboutContentMissingLanguageCode()
  {
    $controller = new AboutController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter for aboutContent.
   */
  public function testAboutContentEmptyLanguageCode()
  {
    $controller = new AboutController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with aboutContent data in requested language.
   */
  public function testAboutContentSuccessFlow()
  {
    // Mock database result row
    $row = (object) [
      'nid' => '10',
      'content' => 'This is about us content.',
      'heading' => 'About Us',
    ];

    // Create a statement mock that supports fetch()
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetch')
      ->willReturn($row);

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('range')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About Content data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEquals('10', $data['data']['nid']);
    $this->assertEquals('en', $data['data']['langcode']);
    $this->assertEquals('English', $data['data']['language_name']);
    $this->assertEquals('About Us', $data['data']['about_heading']);
    $this->assertEquals('This is about us content.', $data['data']['about_content']);
  }

  /**
   * Test case: Success flow with fallback to English when content missing.
   */
  public function testAboutContentFallbackToEnglish()
  {
    // Mock database result row for requested language (empty content/heading)
    $rowEmpty = (object) [
      'nid' => '10',
      'content' => '',
      'heading' => '',
    ];

    // Mock database result row for English fallback
    $rowEnglish = (object) [
      'nid' => '10',
      'content' => 'This is about us content in English.',
      'heading' => 'About Us',
    ];

    // Create statement mocks
    $statementEmpty = $this->createMock(StatementInterface::class);
    $statementEmpty->method('fetch')
      ->willReturn($rowEmpty);

    $statementEnglish = $this->createMock(StatementInterface::class);
    $statementEnglish->method('fetch')
      ->willReturn($rowEnglish);

    // Mock select queries - first call returns empty, second returns English
    $query1 = $this->createMock(SelectInterface::class);
    $query1->method('fields')->willReturnSelf();
    $query1->method('condition')->willReturnSelf();
    $query1->method('range')->willReturnSelf();
    $query1->method('leftJoin')->willReturnSelf();
    $query1->method('addField')->willReturnSelf();
    $query1->method('addExpression')->willReturnSelf();
    $query1->method('execute')->willReturn($statementEmpty);

    $query2 = $this->createMock(SelectInterface::class);
    $query2->method('fields')->willReturnSelf();
    $query2->method('condition')->willReturnSelf();
    $query2->method('range')->willReturnSelf();
    $query2->method('leftJoin')->willReturnSelf();
    $query2->method('addField')->willReturnSelf();
    $query2->method('addExpression')->willReturnSelf();
    $query2->method('execute')->willReturn($statementEnglish);

    // Mock database connection - return different queries on subsequent calls
    $database = $this->createMock(Connection::class);
    $database->expects($this->exactly(2))
      ->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturnOnConsecutiveCalls($query1, $query2);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'fr'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About Content data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEquals('10', $data['data']['nid']);
    $this->assertEquals('en', $data['data']['langcode']);
    $this->assertEquals('English', $data['data']['language_name']);
    $this->assertEquals('About Us', $data['data']['about_heading']);
    $this->assertEquals('This is about us content in English.', $data['data']['about_content']);
  }

  /**
   * Test case: No content found for requested or fallback language.
   */
  public function testAboutContentNoContentFound()
  {
    // Create statement mock that returns FALSE (no rows)
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetch')
      ->willReturn(FALSE);

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('range')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection - return same query for both calls
    $database = $this->createMock(Connection::class);
    $database->expects($this->exactly(2))
      ->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'fr'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(404, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('No about content found for the requested or fallback language.', $data['message']);
  }

  /**
   * Test case: Exception handling for aboutContent.
   */
  public function testAboutContentExceptionHandling()
  {
    // Mock database connection that throws exception
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->willThrowException(new \Exception('Database connection failed'));

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
    $this->assertStringContainsString('Database connection failed', $data['message']);
  }

  /**
   * Test case: Success flow with only heading (content empty).
   */
  public function testAboutContentWithOnlyHeading()
  {
    // Mock database result row with only heading
    $row = (object) [
      'nid' => '15',
      'content' => '',
      'heading' => 'About Our Company',
    ];

    // Create a statement mock that supports fetch()
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetch')
      ->willReturn($row);

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('range')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('Spanish');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('es')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'es'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About Content data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEquals('15', $data['data']['nid']);
    $this->assertEquals('es', $data['data']['langcode']);
    $this->assertEquals('Spanish', $data['data']['language_name']);
    $this->assertEquals('About Our Company', $data['data']['about_heading']);
    $this->assertEquals('', $data['data']['about_content']);
  }

  /**
   * Test case: Language manager returns NULL for invalid language.
   */
  public function testAboutContentInvalidLanguage()
  {
    // Mock database result row
    $row = (object) [
      'nid' => '20',
      'content' => 'Test content',
      'heading' => 'Test heading',
    ];

    // Create a statement mock that supports fetch()
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetch')
      ->willReturn($row);

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('range')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager returning NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/about-content',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutContent($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About Content data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEquals('invalid', $data['data']['language_name']);
  }

  /**
   * Test case: Missing language code parameter for aboutCapabilities.
   */
  public function testAboutCapabilitiesMissingLanguageCode()
  {
    $controller = new AboutController();
    $request = new Request([], [], [], [], [], [], json_encode([]));

    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Empty language code parameter for aboutCapabilities.
   */
  public function testAboutCapabilitiesEmptyLanguageCode()
  {
    $controller = new AboutController();
    $request = new Request([], [], [], [], [], [], json_encode(['lan' => '']));

    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    $this->assertEquals(400, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertEquals('Language code (lan) is required', $data['message']);
  }

  /**
   * Test case: Success flow with capabilities data including files.
   */
  public function testAboutCapabilitiesSuccessFlow()
  {
    // Mock database result rows
    $row1 = (object) [
      'nid' => '1',
      'description' => 'First capability description',
      'heading' => 'First Capability',
      'icon' => '10',
      'image' => '11',
      'link' => 'https://example.com/capability1',
    ];
    $row2 = (object) [
      'nid' => '2',
      'description' => 'Second capability description',
      'heading' => 'Second Capability',
      'icon' => '20',
      'image' => '21',
      'link' => 'https://example.com/capability2',
    ];

    // Create an iterable statement mock
    $statement = new class($row1, $row2) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1, $row2) {
        $this->rows = [$row1, $row2];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock file entities
    $file1 = $this->createMock(FileInterface::class);
    $file1->method('getFileUri')->willReturn('public://icon1.png');

    $file2 = $this->createMock(FileInterface::class);
    $file2->method('getFileUri')->willReturn('public://image1.jpg');

    $file3 = $this->createMock(FileInterface::class);
    $file3->method('getFileUri')->willReturn('public://icon2.png');

    $file4 = $this->createMock(FileInterface::class);
    $file4->method('getFileUri')->willReturn('public://image2.jpg');

    // Mock file storage to handle File::load() calls
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')
      ->willReturnCallback(function ($fid) use ($file1, $file2, $file3, $file4) {
        $fileMap = [
          '10' => $file1,
          '11' => $file2,
          '20' => $file3,
          '21' => $file4,
        ];
        return $fileMap[$fid] ?? NULL;
      });

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository (needed for File::load() static method)
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('file_url_generator', $fileUrlGenerator);
    $container->set('language_manager', $languageManager);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-about-capabilities-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About capabilities data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(2, $data['data']);

    // Check first capability data
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('en', $data['data'][0]['langcode']);
    $this->assertEquals('English', $data['data'][0]['language_name']);
    $this->assertEquals('First capability description', $data['data'][0]['description']);
    $this->assertEquals('First Capability', $data['data'][0]['heading']);
    $this->assertEquals('https://example.com/image1.jpg', $data['data'][0]['contentImg_url']);
    $this->assertEquals('https://example.com/icon1.png', $data['data'][0]['contentIcon']);
    $this->assertEquals('https://example.com/capability1', $data['data'][0]['link']);

    // Check second capability data
    $this->assertEquals('2', $data['data'][1]['nid']);
    $this->assertEquals('en', $data['data'][1]['langcode']);
    $this->assertEquals('English', $data['data'][1]['language_name']);
    $this->assertEquals('Second capability description', $data['data'][1]['description']);
    $this->assertEquals('Second Capability', $data['data'][1]['heading']);
    $this->assertEquals('https://example.com/image2.jpg', $data['data'][1]['contentImg_url']);
    $this->assertEquals('https://example.com/icon2.png', $data['data'][1]['contentIcon']);
    $this->assertEquals('https://example.com/capability2', $data['data'][1]['link']);
  }

  /**
   * Test case: Success flow with no capabilities data found.
   */
  public function testAboutCapabilitiesNoContentFound()
  {
    // Create an iterable statement mock with no rows
    $statement = new class implements StatementInterface, \Iterator {
      private $rows = [];
      private $position = -1;
      private $currentRow = NULL;

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return 0;
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return [];
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        return FALSE;
      }

      public function fetchField($index = 0) {
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return [];
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        return [];
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        return [];
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('file_url_generator', $fileUrlGenerator);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-about-capabilities-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About capabilities data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertEmpty($data['data']);
  }

  /**
   * Test case: Success flow with capabilities data but no files.
   */
  public function testAboutCapabilitiesNoFiles()
  {
    // Mock database result row without file IDs
    $row1 = (object) [
      'nid' => '1',
      'description' => 'Capability without files',
      'heading' => 'No Files Capability',
      'icon' => NULL,
      'image' => NULL,
      'link' => 'https://example.com/capability',
    ];

    // Create an iterable statement mock
    $statement = new class($row1) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1) {
        $this->rows = [$row1];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('French');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('fr')
      ->willReturn($language);

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('file_url_generator', $fileUrlGenerator);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-about-capabilities-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'fr'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About capabilities data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('fr', $data['data'][0]['langcode']);
    $this->assertEquals('French', $data['data'][0]['language_name']);
    $this->assertEquals('Capability without files', $data['data'][0]['description']);
    $this->assertEquals('No Files Capability', $data['data'][0]['heading']);
    $this->assertEquals('', $data['data'][0]['contentImg_url']);
    $this->assertEquals('', $data['data'][0]['contentIcon']);
    $this->assertEquals('https://example.com/capability', $data['data'][0]['link']);
  }

  /**
   * Test case: Success flow with file that cannot be loaded.
   */
  public function testAboutCapabilitiesFileLoadFailure()
  {
    // Mock database result row with file IDs that don't exist
    $row1 = (object) [
      'nid' => '1',
      'description' => 'Capability with invalid file',
      'heading' => 'Invalid File Capability',
      'icon' => '999',
      'image' => '998',
      'link' => 'https://example.com/capability',
    ];

    // Create an iterable statement mock
    $statement = new class($row1) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1) {
        $this->rows = [$row1];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock file storage that returns NULL for invalid file IDs
    $fileStorage = $this->createMock(EntityStorageInterface::class);
    $fileStorage->method('load')
      ->willReturn(NULL);

    // Mock entity type manager
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('file')
      ->willReturn($fileStorage);

    // Mock entity type repository
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $entityTypeRepository->method('getEntityTypeFromClass')
      ->willReturn('file');

    // Mock language
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getName')->willReturn('English');

    // Mock language manager
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('en')
      ->willReturn($language);

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('entity_type.manager', $entityTypeManager);
    $container->set('entity_type.repository', $entityTypeRepository);
    $container->set('language_manager', $languageManager);
    $container->set('file_url_generator', $fileUrlGenerator);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-about-capabilities-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertEquals('About capabilities data fetched successfully', $data['message']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('1', $data['data'][0]['nid']);
    $this->assertEquals('', $data['data'][0]['contentImg_url']);
    $this->assertEquals('', $data['data'][0]['contentIcon']);
  }

  /**
   * Test case: Language manager returns NULL for invalid language.
   */
  public function testAboutCapabilitiesInvalidLanguage()
  {
    // Mock database result row
    $row1 = (object) [
      'nid' => '1',
      'description' => 'Test capability',
      'heading' => 'Test Heading',
      'icon' => NULL,
      'image' => NULL,
      'link' => 'https://example.com/test',
    ];

    // Create an iterable statement mock
    $statement = new class($row1) implements StatementInterface, \Iterator {
      private $rows;
      private $position = -1;
      private $currentRow = NULL;

      public function __construct($row1) {
        $this->rows = [$row1];
      }

      public function execute($args = [], $options = []) {
        return $this;
      }

      public function getQueryString() {
        return '';
      }

      public function getConnectionTarget(): string {
        return 'default';
      }

      public function rowCount() {
        return count($this->rows);
      }

      public function setFetchMode($fetch_mode, $a1 = NULL, $a2 = []) {
        return TRUE;
      }

      public function fetch($fetch_mode = NULL, $cursor_orientation = NULL, $cursor_offset = NULL) {
        $this->position++;
        if ($this->position < count($this->rows)) {
          $this->currentRow = $this->rows[$this->position];
          return $this->currentRow;
        }
        $this->currentRow = FALSE;
        return FALSE;
      }

      public function fetchAll($fetch_mode = NULL, $fetch_column = NULL, $constructor_args = NULL) {
        return $this->rows;
      }

      public function fetchObject($class_name = NULL, $constructor_args = []) {
        return $this->fetch();
      }

      public function fetchAssoc() {
        $row = $this->fetch();
        return $row ? (array) $row : FALSE;
      }

      public function fetchField($index = 0) {
        $row = $this->fetch();
        if ($row) {
          $arr = (array) $row;
          return isset($arr[$index]) ? $arr[$index] : reset($arr);
        }
        return FALSE;
      }

      public function fetchCol($index = 0) {
        return array_column($this->rows, $index);
      }

      public function fetchAllKeyed($key_index = 0, $value_index = 1) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $keys = array_values($arr);
          $result[$keys[$key_index] ?? ''] = $keys[$value_index] ?? '';
        }
        return $result;
      }

      public function fetchAllAssoc($key, $fetch = NULL) {
        $result = [];
        foreach ($this->rows as $row) {
          $arr = (array) $row;
          $result[$arr[$key] ?? ''] = $row;
        }
        return $result;
      }

      // Iterator methods
      public function rewind(): void {
        $this->position = -1;
        $this->currentRow = NULL;
      }

      public function current(): mixed {
        return $this->currentRow;
      }

      public function key(): mixed {
        return $this->position;
      }

      public function next(): void {
        $this->fetch();
      }

      public function valid(): bool {
        if ($this->position === -1) {
          $this->fetch();
        }
        return $this->currentRow !== FALSE && $this->currentRow !== NULL;
      }
    };

    // Mock select query
    $query = $this->createMock(SelectInterface::class);
    $query->method('fields')->willReturnSelf();
    $query->method('leftJoin')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    // Mock database connection
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->with('node_field_data', 'nfd')
      ->willReturn($query);

    // Mock language manager returning NULL for invalid language
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getLanguage')
      ->with('invalid')
      ->willReturn(NULL);

    // Mock file URL generator
    $fileUrlGenerator = $this->createMock(FileUrlGeneratorInterface::class);
    $fileUrlGenerator->method('generateAbsoluteString')
      ->willReturnCallback(function ($uri) {
        return 'https://example.com/' . str_replace('public://', '', $uri);
      });

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);
    $container->set('language_manager', $languageManager);
    $container->set('file_url_generator', $fileUrlGenerator);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-about-capabilities-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'invalid'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertTrue($data['status']);
    $this->assertIsArray($data['data']);
    $this->assertCount(1, $data['data']);
    $this->assertEquals('invalid', $data['data'][0]['language_name']);
  }

  /**
   * Test case: Exception handling for aboutCapabilities.
   */
  public function testAboutCapabilitiesExceptionHandling()
  {
    // Mock database connection that throws exception
    $database = $this->createMock(Connection::class);
    $database->method('select')
      ->willThrowException(new \Exception('Database connection failed'));

    // Register services in container
    $container = \Drupal::getContainer();
    $container->set('database', $database);

    \Drupal::setContainer($container);

    // Make request
    $request = Request::create(
      '/api/get-about-capabilities-data',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      json_encode(['lan' => 'en'])
    );

    // Call controller
    $controller = new AboutController();
    $response = $controller->aboutCapabilities($request);
    $data = json_decode($response->getContent(), TRUE);

    // Assertions
    $this->assertEquals(500, $response->getStatusCode());
    $this->assertFalse($data['status']);
    $this->assertStringContainsString('An error occurred', $data['message']);
    $this->assertStringContainsString('Database connection failed', $data['message']);
  }
}

