<?php

namespace Drupal\api_module\Controller\products;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;
use Exception;
use PhpParser\Node\Stmt\Catch_;
use Drupal\Core\Language\LanguageInterface;

class ProductController extends ControllerBase
{

  /* ####### ########### START PRODUCT SECTION ################################################## */




  public function getProductListingBanner(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 15);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getAllTherapeuticCategory(Request $request)
  {
    try {
      $data = json_decode($request->getContent(), TRUE);
      $requested_langcode = $data['lan'] ?? 'en';

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }
      $database = \Drupal::database();

      // Helper function to fetch taxonomy terms in specific language.
      $fetchTerms = function ($langcode) use ($database) {
        $query = $database->select('taxonomy_term_field_data', 'td')
          ->fields('td', ['tid', 'name', 'langcode']);

        $query->leftJoin('taxonomy_term__field_category_link', 'link', 'link.entity_id = td.tid AND link.langcode = td.langcode');
        $query->addField('link', 'field_category_link_value', 'category_link');

        $query->condition('td.vid', 'therapeutic_category');
        $query->condition('td.status', 1);
        $query->condition('td.langcode', $langcode);

        $query->orderBy('td.name', 'ASC');

        return $query->execute()->fetchAll();
      };

      $results = $fetchTerms($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $results = $fetchTerms('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      // Prepare response data
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        $response_data[] = [
          'tid' => (int) $row->tid,
          'name' => (string) $row->name,
          'category_link' => $row->category_link ?? '',
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Category list fetched successfully.',
        'data' => $response_data,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  } // product Listing page


  public function getAllDosageForm(Request $request)
  {
    try {
      $data = json_decode($request->getContent(), TRUE);
      $requested_langcode = $data['lan'] ?? 'en';

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }
      $database = \Drupal::database();

      // Helper function to fetch dosage terms in a specific language
      $fetchTerms = function ($langcode) use ($database) {
        $query = $database->select('taxonomy_term_field_data', 'td')
          ->fields('td', ['tid', 'name', 'langcode']);

        $query->leftJoin('taxonomy_term__field_dose_form_link', 'link', 'link.entity_id = td.tid AND link.langcode = td.langcode');
        $query->addField('link', 'field_dose_form_link_value', 'dosage_link');

        $query->condition('td.vid', 'dose_form');
        $query->condition('td.status', 1);
        $query->condition('td.langcode', $langcode);

        $query->orderBy('td.name', 'ASC');

        return $query->execute()->fetchAll();
      };

      $results = $fetchTerms($requested_langcode);
      $fallback = false;

      // Fallback to English if no results in requested language
      if (empty($results) && $requested_langcode !== 'en') {
        $results = $fetchTerms('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      // Get readable language name
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        $response_data[] = [
          'tid' => (int)$row->tid,
          'name' => (string)$row->name,
          'dosage_link' => $row->dosage_link ?? '',
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Dosage data fetched successfully.',
        'data' => $response_data,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getAllDevelopmentStatus(Request $request)
  {
    try {
      $data = json_decode($request->getContent(), TRUE);
      $requested_langcode = $data['lan'] ?? 'en';

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }
      $database = \Drupal::database();

      // Helper to fetch dev status terms in specified language
      $fetchTerms = function ($langcode) use ($database) {
        $query = $database->select('taxonomy_term_field_data', 'td')
          ->fields('td', ['tid', 'name', 'langcode']);

        $query->leftJoin('taxonomy_term__field_dev_status_link', 'link', 'link.entity_id = td.tid AND link.langcode = td.langcode');
        $query->addField('link', 'field_dev_status_link_value', 'status_link');

        $query->condition('td.vid', 'dr_reddy_s_development_status');
        $query->condition('td.status', 1);
        $query->condition('td.langcode', $langcode);

        $query->orderBy('td.name', 'ASC');
        return $query->execute()->fetchAll();
      };

      $results = $fetchTerms($requested_langcode);
      $fallback = false;

      // Fallback to English if no results found
      if (empty($results) && $requested_langcode !== 'en') {
        $results = $fetchTerms('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        $response_data[] = [
          'tid' => (int) $row->tid,
          'name' => (string) $row->name,
          'status_link' => $row->status_link ?? '',
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Development status data fetched successfully.',
        'data' => $response_data,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getAllApiTechnology(Request $request)
  {
    try {
      $data = json_decode($request->getContent(), TRUE);
      $requested_langcode = $data['lan'] ?? 'en';

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }
      $database = \Drupal::database();

      // Helper to fetch API technology terms for a specific language.
      $fetchTerms = function ($langcode) use ($database) {
        $query = $database->select('taxonomy_term_field_data', 'td')
          ->fields('td', ['tid', 'name', 'langcode']);

        $query->leftJoin('taxonomy_term__field_api_technology_link', 'link', 'link.entity_id = td.tid AND link.langcode = td.langcode');
        $query->addField('link', 'field_api_technology_link_value', 'api_link');

        $query->condition('td.vid', 'api_technology');
        $query->condition('td.status', 1);
        $query->condition('td.langcode', $langcode);

        $query->orderBy('td.name', 'ASC');
        return $query->execute()->fetchAll();
      };

      $results = $fetchTerms($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $results = $fetchTerms('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        $response_data[] = [
          'tid' => (int)$row->tid,
          'name' => (string)$row->name,
          'api_link' => isset($row->api_link) ? (string)$row->api_link : '',
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'API Technology data fetched successfully.',
        'data' => $response_data,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  /* ####### ########### END OF THE SECTION ################################################## */

  public function getProductDetailsbanner(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $product_link = $data['product_link'];
      $requested_langcode = $data['lan'];

      if (empty($product_link) || empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($product_link) ? 'Product Link parameter is required.' : 'Please specify the language',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode, $product_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'product_details')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->leftJoin('node__field_product_updates', 'nfpt', 'nfpt.entity_id = nfd.nid');
        $query->leftJoin('taxonomy_term__field_menu_link', 'tfl', 'tfl.entity_id = nfpt.field_product_updates_target_id');
        $query->condition('tfl.field_menu_link_value', $product_link);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchData($requested_langcode, $product_link);
      $fallback = false;


      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchData($requested_langcode, $product_link);
        $fallback = true;
      }

      $nodes = Node::loadMultiple($nids);

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      $response_data = [];
      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $title = $node->getTitle();
        $cas_no = $node->get('field_cas_no')->value ?? '';
        $banner_image_id = $node->get('field_product_banner_image')->target_id ?? '';
        $banner_brochure_id = $node->get('field_brochure_link')->target_id ?? '';


        $banner_img = "";
        $brochure_link = "";
        if (!empty($banner_image_id)) {
          $file1 = \Drupal\file\Entity\File::load($banner_image_id);
          if ($file1) {
            $banner_img = \Drupal::service('file_url_generator')->generateAbsoluteString($file1->getFileUri());
          }
        }
        if (!empty($banner_brochure_id)) {
          $file2 = \Drupal\file\Entity\File::load($banner_brochure_id);
          if ($file2) {
            $brochure_link = \Drupal::service('file_url_generator')->generateAbsoluteString($file2->getFileUri());
          }
        }
        $response_data = [
          'nid' => $nid,
          'title' => $title ?? '',
          'banner_image' => $banner_img ?? "",
          'banner_download_brochure_link' => $brochure_link ?? "",
          'product_name' => $row->product_name ?? '',
          'cas_no' => $cas_no ?? '',
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Banner data fetched successfully for Product details page',
        'data' => $response_data,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  /*  public function getProductDetailsAboutContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);
      $product_link = $data['product_link'] ?? NULL;
      $requested_langcode = $data['lan'] ?? 'en';

      if (!$product_link) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Product Link parameter is missing',
        ], 400);
      }

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Step 1: Fetch product node by language
      $fetchNodeData = function ($langcode) use ($database, $product_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title']);

        $query->leftJoin('node__field_product_image', 'img', 'img.entity_id = nfd.nid AND img.delta = 0 AND img.langcode = :langcode');
        $query->addField('img', 'field_product_image_target_id', 'product_img');

        $query->leftJoin('node__field_product_link', 'p_link', 'p_link.entity_id = nfd.nid AND p_link.delta = 0 AND p_link.langcode = :langcode');
        $query->condition('p_link.field_product_link_value', $product_link);
        $query->condition('nfd.type', 'product_details');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->range(0, 1);

        return $query->execute()->fetchObject();
      };

      $node_data = $fetchNodeData($requested_langcode);
      $fallback = false;

      if (!$node_data && $requested_langcode !== 'en') {
        $node_data = $fetchNodeData('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      if (!$node_data) {
        return new JsonResponse([
          'status' => true,
          'data'=>[],
          'message' => 'No matching product found',
        ], 404);
      }

      $nid = $node_data->nid;

      // Step 2: Generate image URL
      $product_image_url = '';
      if (!empty($node_data->product_img)) {
        $file = \Drupal\file\Entity\File::load($node_data->product_img);
        if ($file) {
          $product_image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }

      // Step 3: Generic taxonomy fetcher
      $fetchTermNames = function ($table, $field, $nid) use ($database, $requested_langcode) {
        $tids = $database->select($table, 't')
          ->fields('t', [$field])
          ->condition('t.entity_id', $nid)
          ->condition('t.langcode', $requested_langcode)
          ->execute()
          ->fetchCol();

        if (empty($tids) && $requested_langcode !== 'en') {
          $tids = $database->select($table, 't')
            ->fields('t', [$field])
            ->condition('t.entity_id', $nid)
            ->condition('t.langcode', 'en')
            ->execute()
            ->fetchCol();
        }

        $names = [];
        foreach ($tids as $tid) {
          $term = \Drupal\taxonomy\Entity\Term::load($tid);
          if ($term) {
            $names[] = $term->getName();
          }
        }
        return $names;
      };

      // Step 4: Fetch therapeutic_category with link
      $therapeutic_category = [];
      $tc_tids = $database->select('node__field_therapeutic_category', 'tc')
        ->fields('tc', ['field_therapeutic_category_target_id'])
        ->condition('tc.entity_id', $nid)
        ->condition('tc.langcode', $requested_langcode)
        ->execute()
        ->fetchCol();

      if (empty($tc_tids) && $requested_langcode !== 'en') {
        $tc_tids = $database->select('node__field_therapeutic_category', 'tc')
          ->fields('tc', ['field_therapeutic_category_target_id'])
          ->condition('tc.entity_id', $nid)
          ->condition('tc.langcode', 'en')
          ->execute()
          ->fetchCol();
      }

      if (!empty($tc_tids)) {
        $query = $database->select('taxonomy_term_field_data', 'td')
          ->fields('td', ['tid', 'name']);
        $query->leftJoin('taxonomy_term__field_category_link', 'link', 'link.entity_id = td.tid AND link.langcode = :langcode', [':langcode' => $requested_langcode]);
        $query->addField('link', 'field_category_link_value', 'category_link');
        $query->condition('td.tid', $tc_tids, 'IN');
        $query->distinct();

        $results = $query->execute()->fetchAll();

        foreach ($results as $term) {
          $therapeutic_category[] = [
            'name' => $term->name,
            'category_link' => $term->category_link ?? '',
          ];
        }
      }

      // Step 5: Other fields
      $dose_type = $fetchTermNames('node__field_dose_type', 'field_dose_type_target_id', $nid);
      $api_technology = $fetchTermNames('node__field_api_technology', 'field_api_technology_target_id', $nid);
      $development_stage = $fetchTermNames('node__field_dr_reddy_s_development_sta', 'field_dr_reddy_s_development_sta_target_id', $nid);
      $available_regulatory_filing = $fetchTermNames('node__field_available_regulatory_filin', 'field_available_regulatory_filin_target_id', $nid);

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'About content fetched successfully',
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
        'data' => [
          'nid' => $nid,
          'title' => $node_data->title,
          'product_image_url' => $product_image_url,
          'dose_type' => $dose_type,
          'therapeutic_category' => $therapeutic_category,
          'api_technology' => $api_technology,
          'development_stage' => $development_stage,
          'available_regulatory_filing' => $available_regulatory_filing,
        ],
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }
 */

  public function getProductDetailsAboutContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);
      $product_link = $data['product_link'] ?? NULL;
      $requested_langcode = $data['lan'] ?? 'en';

      if (!$product_link) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Product Link parameter is missing',
        ], 400);
      }

      if (!$requested_langcode) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();
      $langcode = $requested_langcode;

      // Step 1: Fetch node in requested language
      $fetchNodeData = function ($langcode) use ($database, $product_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title']);
        $query->leftJoin(
          'node__field_product_image',
          'img',
          'img.entity_id = nfd.nid AND img.delta = 0 AND img.langcode = :langcode'
        );
        $query->addField('img', 'field_product_image_target_id', 'product_img');
        $query->leftJoin(
          'node__field_product_link',
          'p_link',
          'p_link.entity_id = nfd.nid AND p_link.delta = 0 AND p_link.langcode = :langcode'
        );
        $query->condition('p_link.field_product_link_value', $product_link);
        $query->condition('nfd.type', 'product_details');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->range(0, 1);
        return $query->execute()->fetchObject();
      };

      $node_data = $fetchNodeData($requested_langcode);
      $fallback = false;

      if (!$node_data && $requested_langcode !== 'en') {
        $node_data = $fetchNodeData('en');
        $langcode = 'en';
        $fallback = true;
      }

      if (!$node_data) {
        return new JsonResponse([
          'status' => true,
          'data' => [],
          'message' => 'No matching product found',
        ], 404);
      }

      $nid = $node_data->nid;
      $node_entity = \Drupal\node\Entity\Node::load($nid);

      $available_translations = $node_entity->getTranslationLanguages();
      $available_langcodes = array_keys($available_translations);

      if ($node_entity->hasTranslation($requested_langcode)) {
        $node_entity = $node_entity->getTranslation($requested_langcode);
        $langcode = $requested_langcode;
        $fallback = false;
      }

      // Step 2: Product Image
      $product_image_url = '';
      if ($node_entity->hasField('field_product_image') && !$node_entity->get('field_product_image')->isEmpty()) {
        $file = $node_entity->get('field_product_image')->entity;
        if ($file) {
          $product_image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }

      // Step 3: Helper to fetch translated taxonomy terms
      $fetchTermLabels = function ($field_name) use ($node_entity, $requested_langcode) {
        $labels = [];
        if ($node_entity->hasField($field_name)) {
          foreach ($node_entity->get($field_name)->referencedEntities() as $term) {
            if ($term->hasTranslation($requested_langcode)) {
              $term = $term->getTranslation($requested_langcode);
            } elseif ($term->hasTranslation('en')) {
              $term = $term->getTranslation('en');
            }
            $labels[] = $term->label();
          }
        }
        return $labels;
      };

      // Step 4: Therapeutic category with category_link
      $therapeutic_category = [];
      if ($node_entity->hasField('field_therapeutic_category')) {
        foreach ($node_entity->get('field_therapeutic_category')->referencedEntities() as $term) {
          if ($term->hasTranslation($requested_langcode)) {
            $term = $term->getTranslation($requested_langcode);
          } elseif ($term->hasTranslation('en')) {
            $term = $term->getTranslation('en');
          }

          $category_link = '';
          if ($term->hasField('field_category_link') && !$term->get('field_category_link')->isEmpty()) {
            $category_link = $term->get('field_category_link')->value;
          }

          $therapeutic_category[] = [
            'name' => $term->label(),
            'category_link' => $category_link,
          ];
        }
      }

      // Step 5: Other term fields
      $dose_type = $fetchTermLabels('field_dose_type');
      $api_technology = $fetchTermLabels('field_api_technology');
      $development_stage = $fetchTermLabels('field_dr_reddy_s_development_sta');
      $available_regulatory_filing = $fetchTermLabels('field_available_regulatory_filin');

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$langcode][0] ?? ucfirst($langcode);

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'About content fetched successfully',
        'langcode' => $langcode,
        'language_name' => $language_name,
        'available_translations' => $available_langcodes,
        'data' => [
          'nid' => $nid,
          'title' => $node_entity->label(),
          'product_image_url' => $product_image_url,
          'dose_type' => $dose_type,
          'therapeutic_category' => $therapeutic_category,
          'api_technology' => $api_technology,
          'development_stage' => $development_stage,
          'available_regulatory_filing' => $available_regulatory_filing,
        ],
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }




  public function getProductDetailsDescriptionContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $product_link = $data['product_link'];
      $requested_langcode = $data['lan'];

      if (empty($product_link) || empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($product_link) ? 'Product Link parameter is required.' : 'Please specify the language',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode, $product_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'product_details')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->leftJoin('node__field_product_updates', 'nfpt', 'nfpt.entity_id = nfd.nid');
        $query->leftJoin('taxonomy_term__field_menu_link', 'tfl', 'tfl.entity_id = nfpt.field_product_updates_target_id');
        $query->condition('tfl.field_menu_link_value', $product_link);
        return $query->execute()->fetchCol();
      };
      $nids = $fetchData($requested_langcode, $product_link);
      $fallback = false;


      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchData($requested_langcode, $product_link);
        $fallback = true;
      }

      $nodes = Node::loadMultiple($nids);

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;


      if (empty($nodes)) {
        return new JsonResponse([
          'status' => true,
          'data' => [],
          'message' => 'No matching product found',
        ], 404);
      }


      $response_data = [];
      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $title = $node->getTitle();
        $description = $node->get('field_product_about_text')->value ?? '';

        $response_data = [
          'nid' => $nid,
          'title' => $title ?? '',
          'description' => $description ?? ""
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Product description fetched successfully',
        'langcode' => $requested_langcode,
        'data' => $response_data
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  // public function getProductResourceContent(Request $request)
  // {
  //   try {
  //     $content = $request->getContent();
  //     $data = json_decode($content, TRUE);

  //     $product_link = $data['product_link'] ?? NULL;
  //     $requested_langcode = $data['lan'] ?? 'en';

  //     if (!$product_link) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Product Link parameter is missing',
  //       ], 400);
  //     }

  //     if (!$data['lan']) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'language parameter is missing',
  //       ], 400);
  //     }
  //     $database = \Drupal::database();

  //     // Step 1: Fetch node in requested language (fallback to 'en' if not found)
  //     $fetchNode = function ($langcode) use ($database, $product_link) {
  //       $query = $database->select('node_field_data', 'nfd')
  //         ->fields('nfd', ['nid', 'title']);

  //       $query->leftJoin('node__field_product_link', 'p_link', 'p_link.entity_id = nfd.nid AND p_link.delta = 0 AND p_link.langcode = :langcode');
  //       $query->condition('p_link.field_product_link_value', $product_link);
  //       $query->condition('nfd.type', 'product_details');
  //       $query->condition('nfd.status', 1);
  //       $query->condition('nfd.langcode', $langcode);
  //       $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
  //       $query->range(0, 1);

  //       return $query->execute()->fetchObject();
  //     };

  //     $node_data = $fetchNode($requested_langcode);
  //     $fallback = false;

  //     if (!$node_data && $requested_langcode !== 'en') {
  //       $node_data = $fetchNode('en');
  //       $requested_langcode = 'en';
  //       $fallback = true;
  //     }

  //     if (!$node_data) {
  //       return new JsonResponse([
  //         'status' => true,
  //         'data' => [],
  //         'message' => 'No matching product found',
  //       ], 404);
  //     }

  //     $nid = $node_data->nid;

  //     // Step 2: Define related taxonomy fields
  //     $taxonomyFields = [
  //       'related_articles' => ['table' => 'node__field_related_articles', 'field' => 'field_related_articles_target_id'],
  //       'related_white_papers' => ['table' => 'node__field_related_white_papers', 'field' => 'field_related_white_papers_target_id'],
  //       'related_webinars' => ['table' => 'node__field_related_webinars', 'field' => 'field_related_webinars_target_id'],
  //       'related_news' => ['table' => 'node__field_related_news', 'field' => 'field_related_news_target_id'],
  //       'related_events' => ['table' => 'node__field_related_events', 'field' => 'field_related_events_target_id'],
  //     ];

  //     $related = [];

  //     foreach ($taxonomyFields as $key => $info) {
  //       $tids = $database->select($info['table'], 't')
  //         ->fields('t', [$info['field']])
  //         ->condition('t.entity_id', $nid)
  //         ->condition('t.langcode', $requested_langcode)
  //         ->execute()
  //         ->fetchCol();

  //       // Fallback if no terms found in requested language
  //       if (empty($tids) && $requested_langcode !== 'en') {
  //         $tids = $database->select($info['table'], 't')
  //           ->fields('t', [$info['field']])
  //           ->condition('t.entity_id', $nid)
  //           ->condition('t.langcode', 'en')
  //           ->execute()
  //           ->fetchCol();
  //       }

  //       $names = [];
  //       foreach ($tids as $tid) {
  //         $term = \Drupal\taxonomy\Entity\Term::load($tid);
  //         if ($term) {
  //           $names[] = $term->getName();
  //         }
  //       }
  //       $related[$key] = $names;
  //     }

  //     // Step 3: Return structured response
  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => $fallback
  //         ? 'Fallback to English. No data available in requested language.'
  //         : 'Product resources fetched successfully',
  //       'langcode' => $requested_langcode,
  //       'data' => [
  //         'nid' => $nid,
  //         'title' => $node_data->title,
  //         'related_articles' => $related['related_articles'],
  //         'related_white_papers' => $related['related_white_papers'],
  //         'related_webinars' => $related['related_webinars'],
  //         'related_news' => $related['related_news'],
  //         'related_events' => $related['related_events'],
  //       ],
  //     ]);
  //   } catch (\Throwable $th) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred: ' . $th->getMessage(),
  //     ], 500);
  //   }
  // }
  //updated


  // public function getProductResourceContent(Request $request)
  // {
  //   try {
  //     $content = $request->getContent();
  //     $data = json_decode($content, TRUE);

  //     $product_link = $data['product_link'] ?? '';
  //     $requested_langcode = $data['lan'] ?? '';

  //     if (empty($product_link) || empty($requested_langcode)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => empty($product_link) ? 'Product Link parameter is missing' : 'language parameter is missing',
  //       ], 400);
  //     }

  //     // -------------------------------------------------------------------------
  //     // 1. LOAD PRODUCT NODE + GET ENGLISH menu_link
  //     // -------------------------------------------------------------------------
  //     $node_storage = \Drupal::entityTypeManager()->getStorage('node');
  //     $nids = $node_storage->getQuery()
  //       ->condition('type', 'product_details')
  //       ->condition('field_product_link', $product_link)
  //       ->condition('status', 1)
  //       ->accessCheck(FALSE)
  //       ->execute();

  //     if (empty($nids)) {
  //       return new JsonResponse([
  //         'status' => true,
  //         'message' => 'No matching product found',
  //         'data' => []
  //       ], 200);
  //     }

  //     $node = $node_storage->load(reset($nids));

  //     // English node for English menu_link
  //     $node_en = $node->hasTranslation('en') ? $node->getTranslation('en') : $node;

  //     $nid = $node->hasTranslation($requested_langcode) ? $node->getTranslation($requested_langcode)->id() : $node->id();
  //     $title = $node->hasTranslation($requested_langcode) ? $node->getTranslation($requested_langcode)->label() : $node->label();

  //     // Map: field → resource_id (string)
  //     $field_to_resource_id = [
  //       'field_related_white_papers' => '6',
  //       'field_related_articles'     => '7',
  //       'field_related_news'         => '8',
  //     ];

  //     $english_menu_links = []; // '6' => '/resources/whitepaper-2024'

  //     foreach ($field_to_resource_id as $field => $resource_id_str) {
  //       if ($node_en->get($field)->isEmpty()) continue;

  //       foreach ($node_en->get($field)->referencedEntities() as $term) {
  //         $english_term = $term->hasTranslation('en') ? $term->getTranslation('en') : $term;
  //         $link = $english_term->get('field_menu_link')->value ?? '';
  //         if ($link) {
  //           $english_menu_links[$resource_id_str] = $link;
  //           break;
  //         }
  //       }
  //     }

  //     if (empty($english_menu_links)) {
  //       return new JsonResponse([
  //         'status' => true,
  //         'message' => 'No related resources configured',
  //         'langcode' => $requested_langcode,
  //         'data' => []
  //       ], 200);
  //     }

  //     // -------------------------------------------------------------------------
  //     // 2. CALL METHOD — EXTRACT DATA SAFELY
  //     // -------------------------------------------------------------------------
  //     $related_data = [
  //       'related_articles'     => [],
  //       'related_white_papers' => [],
  //       'related_news'         => [],
  //       'related_events'       => [],
  //       'related_webinars'     => [],
  //     ];

  //     foreach ($english_menu_links as $resource_id_str => $english_link) {
  //       $response = $this->yearWiseResourceList($requested_langcode, $resource_id_str, $english_link);

  //       if (!($response instanceof JsonResponse)) {
  //         continue;
  //       }

  //       $content = $response->getContent();
  //       $json = json_decode($content, true);

  //       if (!is_array($json) || empty($json['status'])) {
  //         continue;
  //       }

  //       $result_data = $json['data'] ?? null;

  //       // $result_data is a SINGLE entry, NOT array[0]
  //       if (!is_array($result_data) || empty($result_data)) {
  //         continue;
  //       }

  //       // USE $result_data DIRECTLY — NO [0]
  //       $entry = $result_data;

  //       $item = [
  //         'title'       => $entry['title'] ?? '',
  //         'url'         => $english_link,
  //         'description' => $entry['description'] ?? '',
  //         'image'       => $entry['feature_image'] ? $entry['feature_image'] : $entry['image'],
  //       ];

  //       $key_map = [
  //         '6' => 'related_white_papers',
  //         '7' => 'related_articles',
  //         '8' => 'related_news',
  //       ];

  //       $key = $key_map[$resource_id_str] ?? null;
  //       if ($key) {
  //         $related_data[$key][] = $item;
  //       }
  //     }

  //     // -------------------------------------------------------------------------
  //     // 3. FINAL RESPONSE
  //     // -------------------------------------------------------------------------
  //     $has_data = !empty($related_data['related_articles'])
  //       || !empty($related_data['related_white_papers'])
  //       || !empty($related_data['related_news']);

  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => $has_data ? 'Product resources fetched successfully' : 'No data found',
  //       'langcode' => $requested_langcode,
  //       'data' => $has_data ? [
  //         'nid' => $nid,
  //         'title' => $title,
  //         'related_articles'     => $related_data['related_articles'],
  //         'related_white_papers' => $related_data['related_white_papers'],
  //         'related_news'         => $related_data['related_news'],
  //         'related_events'       => [],
  //         'related_webinars'     => [],
  //       ] : [],
  //     ]);
  //   } catch (\Throwable $th) {
  //     \Drupal::logger('product_api')->error($th->getMessage());
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred: ' . $th->getMessage(),
  //     ], 500);
  //   }
  // }
  // public function yearWiseResourceList($langcode, $resource_id, $resource_menu_link)
  // {
  //   try {
  //     if (empty($langcode)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Language parameter is missing',
  //       ], 400);
  //     }

  //     // 🔹 Single resource by menu_link
  //     if (!empty($resource_menu_link)) {
  //       $result = [];

  //       $nids = \Drupal::entityQuery('node')
  //         ->condition('type', 'resources_details')
  //         ->condition('status', 1)
  //         ->accessCheck(TRUE)
  //         ->execute();

  //       foreach ($nids as $nid) {
  //         $node = \Drupal\node\Entity\Node::load($nid);
  //         $translated_node = $node->hasTranslation($langcode) ? $node->getTranslation($langcode) : $node;

  //         if ($translated_node->hasField('field_resource_details') && !$translated_node->get('field_resource_details')->isEmpty()) {
  //           foreach ($translated_node->get('field_resource_details') as $yearwise_item) {
  //             $yearwise_para = $yearwise_item->entity;
  //             if (!$yearwise_para || $yearwise_para->getType() !== 'resource_year_wise') continue;

  //             $translated_yearwise_para = $yearwise_para->hasTranslation($langcode)
  //               ? $yearwise_para->getTranslation($langcode)
  //               : $yearwise_para;

  //             if ($translated_yearwise_para->hasField('field_res_details') && !$translated_yearwise_para->get('field_res_details')->isEmpty()) {
  //               foreach ($translated_yearwise_para->get('field_res_details') as $entry_item) {
  //                 $entry_para = $entry_item->entity;
  //                 if (!$entry_para || $entry_para->getType() !== 'resource_details') continue;

  //                 $translated_entry_para = $entry_para->hasTranslation($langcode)
  //                   ? $entry_para->getTranslation($langcode)
  //                   : $entry_para;

  //                 $term = $translated_entry_para->get('field_resource_name')->entity ?? null;
  //                 if ($term) {
  //                   $translated_term = $term->hasTranslation($langcode) ? $term->getTranslation($langcode) : $term;
  //                   $menu_link = $translated_term->get('field_menu_link')->value ?? '';

  //                   if ($menu_link === $resource_menu_link) {
  //                     $result = [
  //                       'title' => $translated_entry_para->get('field_resource_main_heading')->value ?? '',
  //                       'description' => $translated_entry_para->get('field_res_description')->value ?? '',
  //                       'date' => $translated_entry_para->get('field_resource_date')->value ?? '',
  //                       'image' => !$translated_entry_para->get('field_resource_main_image')->isEmpty()
  //                         ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_resource_main_image')->entity->getFileUri())
  //                         : '',
  //                       'feature_image' => !$translated_entry_para->get('field_resource_feat_image')->isEmpty()
  //                         ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_resource_feat_image')->entity->getFileUri())
  //                         : '',
  //                       'banner_image' => !$translated_entry_para->get('field_resource_banner_image')->isEmpty()
  //                         ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_resource_banner_image')->entity->getFileUri())
  //                         : '',
  //                       'download_brochure' => !$translated_entry_para->get('field_download_brochure')->isEmpty()
  //                         ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_download_brochure')->entity->getFileUri())
  //                         : '',
  //                       'resource_type_name' => $translated_term->label(),
  //                       'menu_link' => $menu_link,
  //                     ];


  //                     return new JsonResponse([
  //                       'status' => true,
  //                       'data' => $result,
  //                     ]);
  //                   }
  //                 }
  //               }
  //             }
  //           }
  //         }
  //       }

  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Resource not found for this menu link',
  //       ], 404);
  //     }

  //     // 🔹 Year-wise grouped listing
  //     if (!$resource_id) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Resource ID is missing',
  //       ], 400);
  //     }

  //     $nids = \Drupal::entityQuery('node')
  //       ->condition('type', 'resources_details')
  //       ->condition('status', 1)
  //       ->condition('field_res_type', $resource_id)
  //       ->accessCheck(TRUE)
  //       ->execute();

  //     if (empty($nids)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'No content found for this resource ID',
  //       ], 404);
  //     }

  //     $output = [];

  //     foreach ($nids as $nid) {
  //       $node = \Drupal\node\Entity\Node::load($nid);
  //       $translated_node = $node->hasTranslation($langcode) ? $node->getTranslation($langcode) : $node;

  //       if ($translated_node->hasField('field_resource_details') && !$translated_node->get('field_resource_details')->isEmpty()) {
  //         foreach ($translated_node->get('field_resource_details') as $yearwise_item) {
  //           $yearwise_para = $yearwise_item->entity;
  //           if (!$yearwise_para || $yearwise_para->getType() !== 'resource_year_wise') continue;

  //           $translated_yearwise_para = $yearwise_para->hasTranslation($langcode)
  //             ? $yearwise_para->getTranslation($langcode)
  //             : $yearwise_para;

  //           $year = $translated_yearwise_para->get('field_year')->value ?? '';
  //           $entries = [];

  //           if ($translated_yearwise_para->hasField('field_res_details') && !$translated_yearwise_para->get('field_res_details')->isEmpty()) {
  //             foreach ($translated_yearwise_para->get('field_res_details') as $entry_item) {
  //               $entry_para = $entry_item->entity;
  //               if (!$entry_para || $entry_para->getType() !== 'resource_details') continue;

  //               $translated_entry_para = $entry_para->hasTranslation($langcode)
  //                 ? $entry_para->getTranslation($langcode)
  //                 : $entry_para;

  //               $title = $translated_entry_para->get('field_resource_main_heading')->value ?? '';
  //               $description = $translated_entry_para->get('field_res_description')->value ?? '';
  //               $date = $translated_entry_para->get('field_resource_date')->value ?? '';

  //               $main_image_url = !$translated_entry_para->get('field_resource_main_image')->isEmpty()
  //                 ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_resource_main_image')->entity->getFileUri())
  //                 : '';

  //               $feature_image_url = !$translated_entry_para->get('field_resource_feat_image')->isEmpty()
  //                 ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_resource_feat_image')->entity->getFileUri())
  //                 : '';

  //               $banner_image_url = !$translated_entry_para->get('field_resource_banner_image')->isEmpty()
  //                 ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_resource_banner_image')->entity->getFileUri())
  //                 : '';

  //               $download_brochure = !$translated_entry_para->get('field_download_brochure')->isEmpty()
  //                 ? \Drupal::service('file_url_generator')->generateAbsoluteString($translated_entry_para->get('field_download_brochure')->entity->getFileUri())
  //                 : '';

  //               $resource_type_name = '';
  //               $menu_link = '';
  //               if (!$translated_entry_para->get('field_resource_name')->isEmpty()) {
  //                 $term = $translated_entry_para->get('field_resource_name')->entity;
  //                 if ($term) {
  //                   $translated_term = $term->hasTranslation($langcode) ? $term->getTranslation($langcode) : $term;
  //                   $resource_type_name = $translated_term->label();
  //                   $menu_link = $translated_term->get('field_menu_link')->value ?? '';
  //                 }
  //               }

  //               $entry = [
  //                 'title' => $title,
  //                 'banner_heading' => $title,
  //                 'description' => $description,
  //                 'date' => $date,
  //                 'image' => $main_image_url,
  //                 'feature_image' => $feature_image_url,
  //                 'banner_image' => $banner_image_url,
  //                 'download_brochure' => $download_brochure,
  //                 'resource_type_name' => $resource_type_name,
  //                 'menu_link' => $menu_link,
  //               ];

  //               if ($resource_id == 7 || $resource_id == 8) {
  //                 $entry['year'] = $year;
  //                 $output[] = $entry;
  //               } else {
  //                 $entries[] = $entry;
  //               }
  //             }
  //           }

  //           if (($resource_id != 7 || $resource_id != 8) && ($year != null) && !empty($entries)) {
  //             $output[] = [
  //               'year' => $year,
  //               'resources' => $entries,
  //             ];
  //           }
  //         }
  //       }
  //     }

  //     return new JsonResponse([
  //       'status' => true,
  //       'data' => $output,
  //     ]);
  //   } catch (\Throwable $th) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred: ' . $th->getMessage(),
  //     ], 500);
  //   }
  // }

  public function getProductResourceContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $product_link = $data['product_link'] ?? '';
      $requested_langcode = $data['lan'] ?? '';

      if (empty($product_link) || empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($product_link) ? 'Product Link parameter is missing' : 'language parameter is missing',
        ], 400);
      }

      //loading the product node
      $nids = \Drupal::entityTypeManager()
        ->getStorage('node')
        ->getQuery()
        ->condition('type', 'product_details')
        ->condition('field_product_link', $product_link)
        ->condition('status', 1)
        ->accessCheck(FALSE)
        ->execute();

      if (empty($nids)) {
        return new JsonResponse([
          'status'  => TRUE,
          'message' => 'No matching product found',
          'data'    => [],
        ], 200);
      }

      $node = Node::load(reset($nids));   // there is only one
      $extractMenuLinks = function (array $term_ids) {
        $links = [];
        foreach ($term_ids as $delta) {
          $term = \Drupal::entityTypeManager()
            ->getStorage('taxonomy_term')
            ->load($delta['target_id']);

          if (!$term || !$term->hasField('field_menu_link')) {
            continue;
          }
          if ($term->hasTranslation('en')) {
            $term = $term->getTranslation('en');
          }
          if (!$term->get('field_menu_link')->isEmpty()) {
            $links[] = $term->get('field_menu_link')->value;
          }
        }
        return $links;
      };
      $related_links = [
        'articles'    => $node->hasField('field_related_articles')
          ? $extractMenuLinks($node->get('field_related_articles')->getValue())
          : [],
        'news'        => $node->hasField('field_related_news')
          ? $extractMenuLinks($node->get('field_related_news')->getValue())
          : [],
        'white-paper' => $node->hasField('field_related_white_papers')
          ? $extractMenuLinks($node->get('field_related_white_papers')->getValue())
          : [],
      ];

      $response_data = [
        'nid'                   => $node->id(),
        'title'                 => $node->getTitle(),
        'product_link'          => $node->get('field_product_link')->value,
        'related_articles'      => $this->ProductResHelper($requested_langcode, $related_links['articles'], 'articles'),
        'related_white_papers'  => $this->ProductResHelper($requested_langcode, $related_links['white-paper'], 'white-paper'),
        'related_news'          => $this->ProductResHelper($requested_langcode, $related_links['news'], 'news'),
        'related_events'        => [],
        'related_webinars'      => [],
      ];

      return new JsonResponse([
        'status'  => true,
        'message' => 'Product data retrieved successfully',
        'langcode' => $requested_langcode ?? 'en',
        'data'    => $response_data,
      ], 200);
    } catch (\Throwable $th) {
      \Drupal::logger('product_api')->error($th->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }
  function ProductResHelper($LANGUAGE, $HARD_LINK, $RESOURCE_TYPE)
  {
    // $LANGUAGE = 'en';
    // $HARD_CODED_LINK = ['asia-pacific-climate-leader-award'];

    $nids = \Drupal::entityQuery('node')
      ->condition('type', 'resources_details')
      ->accessCheck(FALSE)
      ->execute();

    $nodes = Node::loadMultiple($nids);
    $resources = [];

    $path_alias_manager = \Drupal::service('path_alias.manager');
    $entity_repo        = \Drupal::service('entity.repository');

    foreach ($nodes as $node) {
      $node = $entity_repo->getTranslationFromContext($node, $LANGUAGE, ['fallback' => TRUE]);

      if ($node->language()->getId() === LanguageInterface::LANGCODE_NOT_SPECIFIED) {
        continue;
      }

      $type_term = $node->get('field_res_type')->entity;
      if (!$type_term) {
        continue;
      }

      $type_term = $entity_repo->getTranslationFromContext($type_term, $LANGUAGE, ['fallback' => TRUE]);
      if (strtolower($type_term->label()) !== $RESOURCE_TYPE) {
        continue;
      }

      foreach ($node->get('field_resource_details') as $ref) {
        $year_para = $ref->entity;
        if (
          !$year_para
          || $year_para->getEntityTypeId() !== 'paragraph'
          || $year_para->bundle() !== 'resource_year_wise'
        ) {
          continue;
        }

        $year_para = $entity_repo->getTranslationFromContext($year_para, $LANGUAGE, ['fallback' => TRUE]);

        foreach ($year_para->get('field_res_details') as $detail_ref) {
          $detail_para = $detail_ref->entity;
          if (!$detail_para || $detail_para->bundle() !== 'resource_details') {
            continue;
          }

          $detail_para = $entity_repo->getTranslationFromContext($detail_para, $LANGUAGE, ['fallback' => TRUE]);
          $resource_url   = '';
          $resource_term  = $detail_para->get('field_resource_name')->entity;
          if ($resource_term && $resource_term->hasField('field_menu_link') && !$resource_term->get('field_menu_link')->isEmpty()) {
            $term_trans = $entity_repo->getTranslationFromContext($resource_term, $LANGUAGE, ['fallback' => TRUE]);
            $term_en    = $entity_repo->getTranslationFromContext($resource_term, 'en', ['fallback' => TRUE]);

            $raw_path_trans = $term_trans->get('field_menu_link')->value;
            $raw_path_en    = $term_en->get('field_menu_link')->value;

            $normalize = function ($raw) {
              if (empty($raw)) {
                return '';
              }
              if (strpos($raw, '/') !== FALSE) {
                return ltrim($raw, '/');
              }
              if (preg_match('/^\D*?(\d+)$/', $raw, $m)) {
                return 'taxonomy/term/' . $m[1];
              }
              return ltrim($raw, '/');
            };

            $internal_en    = $normalize($raw_path_en);
            $internal_trans = $normalize($raw_path_trans);

            $english_alias_raw    = $path_alias_manager->getAliasByPath('/' . $internal_en, 'en');
            $translated_alias_raw = $path_alias_manager->getAliasByPath('/' . $internal_trans, $LANGUAGE);

            $english_path    = is_string($english_alias_raw) ? ltrim(parse_url($english_alias_raw, PHP_URL_PATH), '/') : '';
            $translated_path = is_string($translated_alias_raw) ? ltrim(parse_url($translated_alias_raw, PHP_URL_PATH), '/') : '';

            if (!in_array($english_path, $HARD_LINK, TRUE)) {
              continue;
            }

            $final_alias = (!empty($translated_path) && $translated_path !== $english_path)
              ? $translated_alias_raw
              : $english_alias_raw;

            $resource_url = $final_alias;
          }
          // ------------------------------------

          $title       = $detail_para->get('field_resource_main_heading')->value ?? '';
          $description = $detail_para->hasField('field_res_description')
            && !$detail_para->get('field_res_description')->isEmpty()
            ? strip_tags($detail_para->get('field_res_description')->value)
            : '';

          $image_url = '';

          if (
            $detail_para->hasField('field_resource_feat_image') &&
            !$detail_para->get('field_resource_feat_image')->isEmpty()
          ) {
            // Use featured image first if available
            $file = $detail_para->get('field_resource_feat_image')->entity;
            if ($file) {
              $image_url = \Drupal::service('file_url_generator')
                ->generateAbsoluteString($file->getFileUri());
            }
          } elseif (
            $detail_para->hasField('field_resource_main_image') &&
            !$detail_para->get('field_resource_main_image')->isEmpty()
          ) {
            // Fallback to main image if featured not found
            $file = $detail_para->get('field_resource_main_image')->entity;
            if ($file) {
              $image_url = \Drupal::service('file_url_generator')
                ->generateAbsoluteString($file->getFileUri());
            }
          }
          $url = $resource_url;
          if (str_starts_with($url, '/')) {
            $url = ltrim($url, '/');
          }
          $resources[] = [
            'title'       => $title,
            'description' => $description,
            'url'         => $url,
            'image'       => $image_url,
          ];
        }
      }
    }

    return $resources;
  }




  /*   public function paginateProductList(Request $request) {
    try {
      $data = json_decode($request->getContent(), TRUE);

      $page = isset($data['page']) ? max(1, (int)$data['page']) : 1;
      $limit = isset($data['limit']) ? max(1, (int)$data['limit']) : 10;
      $offset = ($page - 1) * $limit;
      $requested_langcode = $data['lan'] ?? 'en';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $filters = [
        'therapeutic_categories' => $data['therapeutic_categories'] ?? [],
        'dosage_forms' => $data['dosage_forms'] ?? [],
        'development_stages' => $data['development_stages'] ?? [],
        'api_technologies' => $data['api_technologies'] ?? [],
        'starts_with' => $data['starts_with'] ?? '',
      ];

      $filters_applied = !empty($filters['therapeutic_categories']) ||
                        !empty($filters['dosage_forms']) ||
                        !empty($filters['development_stages']) ||
                        !empty($filters['api_technologies']) ||
                        !empty($filters['starts_with']);

      if ($filters_applied) {
        $offset = 0;
        $limit = 10000;
      }

      $database = \Drupal::database();

      // Count function
      $getTotalCount = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd');
        $query->join('node__field_therapeutic_category', 'cat', 'cat.entity_id = nfd.nid AND cat.delta = 0');
        $query->join('node__field_dr_reddy_s_development_sta', 'status', 'status.entity_id = nfd.nid AND status.delta = 0');
        $query->condition('nfd.type', 'product_details');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression('COUNT(DISTINCT nfd.nid)', 'total');
        return $query->execute()->fetchField();
      };

      $total = $getTotalCount($requested_langcode);
      $fallback = false;

      if ($total == 0 && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $total = $getTotalCount($requested_langcode);
        $fallback = true;
      }

      // Set taxonomy langcode to match node language
      $taxonomy_langcode = $requested_langcode;

      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid', 'langcode'])
        ->distinct()
        ->range($offset, $limit);

      $query->join('node__field_product_name', 'pname', 'pname.entity_id = nfd.nid AND pname.delta = 0 AND pname.langcode = nfd.langcode');
      $query->addField('pname', 'field_product_name_value', 'product_name');

      $query->leftJoin('node__field_product_link', 'plink', 'plink.entity_id = nfd.nid AND plink.delta = 0 AND plink.langcode = nfd.langcode');
      $query->addField('plink', 'field_product_link_value', 'product_link');

      // ✅ LEFT JOIN for category
      $query->join('node__field_therapeutic_category', 'cat', 'cat.entity_id = nfd.nid AND cat.delta = 0 AND cat.langcode = nfd.langcode');
      $query->leftJoin('taxonomy_term_field_data', 'cat_term', 'cat_term.tid = cat.field_therapeutic_category_target_id AND cat_term.langcode = :taxonomy_langcode', [':taxonomy_langcode' => $taxonomy_langcode]);
      $query->addField('cat_term', 'name', 'category_name');

      // ✅ LEFT JOIN for development status
      $query->join('node__field_dr_reddy_s_development_sta', 'status', 'status.entity_id = nfd.nid AND status.delta = 0 AND status.langcode = nfd.langcode');
      $query->leftJoin('taxonomy_term_field_data', 'status_term', 'status_term.tid = status.field_dr_reddy_s_development_sta_target_id AND status_term.langcode = :taxonomy_langcode', [':taxonomy_langcode' => $taxonomy_langcode]);
      $query->addField('status_term', 'name', 'status_name');

      // ✅ Dosage form filter
      if (!empty($filters['dosage_forms'])) {
        $query->leftJoin('node__field_dose_type', 'dosage', 'dosage.entity_id = nfd.nid AND dosage.langcode = nfd.langcode');
        $query->leftJoin('taxonomy_term_field_data', 'dosage_term', 'dosage_term.tid = dosage.field_dose_type_target_id AND dosage_term.langcode = :taxonomy_langcode', [':taxonomy_langcode' => $taxonomy_langcode]);
        $query->condition('dosage_term.tid', $filters['dosage_forms'], 'IN');
      }

      // ✅ API technology filter
      if (!empty($filters['api_technologies'])) {
        $query->leftJoin('node__field_api_technology', 'api', 'api.entity_id = nfd.nid AND api.langcode = nfd.langcode');
        $query->leftJoin('taxonomy_term_field_data', 'api_term', 'api_term.tid = api.field_api_technology_target_id AND api_term.langcode = :taxonomy_langcode', [':taxonomy_langcode' => $taxonomy_langcode]);
        $query->condition('api_term.tid', $filters['api_technologies'], 'IN');
      }

      // Core node filters
      $query->condition('nfd.type', 'product_details');
      $query->condition('nfd.status', 1);
      $query->condition('nfd.langcode', $requested_langcode);

      if (!empty($filters['therapeutic_categories'])) {
        $query->condition('cat.field_therapeutic_category_target_id', $filters['therapeutic_categories'], 'IN');
      }

      if (!empty($filters['development_stages'])) {
        $query->condition('status.field_dr_reddy_s_development_sta_target_id', $filters['development_stages'], 'IN');
      }

      if (!empty($filters['starts_with'])) {
        $query->where("LOWER(LEFT(pname.field_product_name_value, 1)) = :letter", [
          ':letter' => strtolower($filters['starts_with'])
        ]);
      }

      $results = $query->execute();

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        $response_data[] = [
          'nid' => (int) $row->nid,
          'product_name' => $row->product_name ?? '',
          'product_link' => $row->product_link ?? '',
          'category_name' => $row->category_name ?? '',
          'development_status' => $row->status_name ?? '',
          'langcode' => $row->langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English. No data in requested language.' : 'Product data fetched successfully.',
        'data' => $response_data,
        'pagination' => $filters_applied ? [] : [
          'total' => (int)$total,
          'page' => $page,
          'limit' => $limit,
          'total_pages' => ceil($total / $limit),
        ],
      ]);

    } catch (\Exception $e) {
      \Drupal::logger('api_module')->error($e->getMessage());
      return new JsonResponse(['error' => 'An error occurred while fetching product data.'], 500);
    }
  } */

  public function paginateProductList(Request $request)
  {
    try {
      $data = json_decode($request->getContent(), TRUE);

      $page = isset($data['page']) ? max(1, (int)$data['page']) : 1;
      $limit = isset($data['limit']) ? max(1, (int)$data['limit']) : 10;
      $offset = ($page - 1) * $limit;
      $requested_langcode = $data['lan'] ?? 'en';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $filters = [
        'therapeutic_categories' => $data['therapeutic_categories'] ?? [],
        'dosage_forms' => $data['dosage_forms'] ?? [],
        'development_stages' => $data['development_stages'] ?? [],
        'api_technologies' => $data['api_technologies'] ?? [],
        'starts_with' => $data['starts_with'] ?? '',
      ];

      $filters_applied = !empty($filters['therapeutic_categories']) ||
        !empty($filters['dosage_forms']) ||
        !empty($filters['development_stages']) ||
        !empty($filters['api_technologies']) ||
        !empty($filters['starts_with']);

      if ($filters_applied) {
        $offset = 0;
        $limit = 10000;
      }

      $database = \Drupal::database();

      // Count function
      $getTotalCount = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd');
        $query->condition('nfd.type', 'product_details');
        $query->condition('nfd.status', 1);

        $query->leftJoin('node__field_hide_frontend', 'hide', 'hide.entity_id = nfd.nid');
        $orGroup = $query->orConditionGroup()
          ->condition('hide.field_hide_frontend_value', '0', '=')
          ->isNull('hide.field_hide_frontend_value');
        $query->condition($orGroup);

        $query->condition('nfd.langcode', $langcode);
        $query->addExpression('COUNT(DISTINCT nfd.nid)', 'total');

        return $query->execute()->fetchField();
      };

      $total = $getTotalCount($requested_langcode);
      $fallback = false;

      if ($total == 0 && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $total = $getTotalCount($requested_langcode);
        $fallback = true;
      }

      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid', 'langcode'])
        ->distinct()
        ->range($offset, $limit);

      $query->join('node__field_product_name', 'pname', 'pname.entity_id = nfd.nid AND pname.delta = 0 AND pname.langcode = nfd.langcode');
      $query->addField('pname', 'field_product_name_value', 'product_name');

      $query->leftJoin('node__field_product_link', 'plink', 'plink.entity_id = nfd.nid AND plink.delta = 0 AND plink.langcode = nfd.langcode');
      $query->addField('plink', 'field_product_link_value', 'product_link');

      // Join therapeutic_category and development_status taxonomy IDs
      $query->leftJoin('node__field_therapeutic_category', 'cat', 'cat.entity_id = nfd.nid AND cat.delta = 0');
      $query->addField('cat', 'field_therapeutic_category_target_id', 'category_tid');

      $query->leftJoin('node__field_dr_reddy_s_development_sta', 'status', 'status.entity_id = nfd.nid AND status.delta = 0');
      $query->addField('status', 'field_dr_reddy_s_development_sta_target_id', 'status_tid');

      // Optional filters
      if (!empty($filters['therapeutic_categories'])) {
        $query->condition('cat.field_therapeutic_category_target_id', $filters['therapeutic_categories'], 'IN');
      }

      if (!empty($filters['development_stages'])) {
        $query->condition('status.field_dr_reddy_s_development_sta_target_id', $filters['development_stages'], 'IN');
      }

      if (!empty($filters['dosage_forms'])) {
        $query->leftJoin('node__field_dose_type', 'dosage', 'dosage.entity_id = nfd.nid AND dosage.langcode = nfd.langcode');
        $query->leftJoin('taxonomy_term_field_data', 'dosage_term', 'dosage_term.tid = dosage.field_dose_type_target_id AND dosage_term.langcode = :taxonomy_langcode', [':taxonomy_langcode' => $requested_langcode]);
        $query->condition('dosage_term.tid', $filters['dosage_forms'], 'IN');
      }

      if (!empty($filters['api_technologies'])) {
        $query->leftJoin('node__field_api_technology', 'api', 'api.entity_id = nfd.nid AND api.langcode = nfd.langcode');
        $query->leftJoin('taxonomy_term_field_data', 'api_term', 'api_term.tid = api.field_api_technology_target_id AND api_term.langcode = :taxonomy_langcode', [':taxonomy_langcode' => $requested_langcode]);
        $query->condition('api_term.tid', $filters['api_technologies'], 'IN');
      }

      $query->condition('nfd.type', 'product_details');
      $query->condition('nfd.status', 1);
      $query->condition('nfd.langcode', $requested_langcode);

      if (!empty($filters['starts_with'])) {
        $query->where("LOWER(LEFT(pname.field_product_name_value, 1)) = :letter", [
          ':letter' => strtolower($filters['starts_with'])
        ]);
      }
      //Filtering logic
      $query->leftJoin('node__field_hide_frontend', 'hide', 'hide.entity_id = nfd.nid');
      $orGroup = $query->orConditionGroup()
        ->condition('hide.field_hide_frontend_value', '0', '=')
        ->isNull('hide.field_hide_frontend_value');
      $query->condition($orGroup);

      $results = $query->execute();

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        // Category name fallback
        $category_name = '';
        if (!empty($row->category_tid)) {
          $term = \Drupal\taxonomy\Entity\Term::load($row->category_tid);
          if ($term) {
            if ($term->hasTranslation($requested_langcode)) {
              $term = $term->getTranslation($requested_langcode);
            } elseif ($requested_langcode !== 'en' && $term->hasTranslation('en')) {
              $term = $term->getTranslation('en');
            }
            $category_name = $term->label();
          }
        }

        // Development status fallback
        $status_name = '';
        if (!empty($row->status_tid)) {
          $term = \Drupal\taxonomy\Entity\Term::load($row->status_tid);
          if ($term) {
            if ($term->hasTranslation($requested_langcode)) {
              $term = $term->getTranslation($requested_langcode);
            } elseif ($requested_langcode !== 'en' && $term->hasTranslation('en')) {
              $term = $term->getTranslation('en');
            }
            $status_name = $term->label();
          }
        }

        $response_data[] = [
          'nid' => (int) $row->nid,
          'product_name' => $row->product_name ?? '',
          'product_link' => $row->product_link ?? '',
          'category_name' => $category_name,
          'development_status' => $status_name,
          'langcode' => $row->langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English. No data in requested language.' : 'Product data fetched successfully.',
        'data' => $response_data,
        'pagination' => $filters_applied ? [] : [
          'total' => (int)$total,
          'page' => $page,
          'limit' => $limit,
          'total_pages' => ceil($total / $limit),
        ],
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('api_module')->error($e->getMessage());
      return new JsonResponse(['error' => 'An error occurred while fetching product data.'], 500);
    }
  }





  public function getallProductName(Request $request)
  {
    try {
      $data = json_decode($request->getContent(), TRUE);
      $requested_langcode = $data['lan'] ?? 'en';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Count check to see if data exists in requested language
      $getTotalCount = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd');
        $query->condition('nfd.type', 'product_details');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression('COUNT(DISTINCT nfd.nid)', 'total');
        return $query->execute()->fetchField();
      };

      $total = $getTotalCount($requested_langcode);
      $fallback = false;

      if ($total == 0 && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $total = $getTotalCount($requested_langcode);
        $fallback = true;
      }

      // Main Query: fetch all product names
      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid', 'langcode'])
        ->distinct();

      $query->join('node__field_product_name', 'pname', 'pname.entity_id = nfd.nid AND pname.delta = 0 AND pname.langcode = nfd.langcode');
      $query->addField('pname', 'field_product_name_value', 'product_name');

      $query->condition('nfd.type', 'product_details');
      $query->condition('nfd.status', 1);
      $query->condition('nfd.langcode', $requested_langcode);

      $results = $query->execute();

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

      $response_data = [];
      foreach ($results as $row) {
        $response_data[] = [
          'nid' => (int) $row->nid,
          'product_name' => $row->product_name ?? '',
          'langcode' => $row->langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English. No data in requested language.' : 'Product names fetched successfully.',
        'data' => $response_data,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('api_module')->error($e->getMessage());
      return new JsonResponse(['error' => 'An error occurred while fetching product names.'], 500);
    }
  } // drop down call all project


  public function getallGenericProductName(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      // Get language name for metadata
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      // Query all published generic formulations nodes
      $nids = \Drupal::entityQuery('node')
        ->condition('type', 'generic_formulations_products')
        ->condition('status', 1)
        ->accessCheck(true)
        ->execute();

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $product_list = [];

      foreach ($nodes as $node) {
        // Language fallback
        if ($node->hasTranslation($langcode)) {
          $node = $node->getTranslation($langcode);
        }

        // Ensure product name field exists and is not empty
        if ($node->hasField('field_product_name_gf') && !$node->get('field_product_name_gf')->isEmpty()) {
          $product_list[] = [
            'nid' => $node->id(),
            'product_name' => $node->get('field_product_name_gf')->value,
          ];
        }
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'Product names retrieved successfully',
        'language' => $language_name,
        'data' => $product_list,
      ], 200);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => $e->getMessage(),
      ], 500);
    }
  } // generic product
  public function  languagewiseProductList(Request $request)
  {
    $data = json_decode($request->getContent(), TRUE);
    $product_link = $data['product_link'] ?? '';
    $requested_lan = $data['requested_lan'] ?? 'en';

    if (empty($product_link)) {
      return new JsonResponse(['error' => 'Product link is required'], 400);
    }

    // Load taxonomy term by field_menu_link (api_products)
    $query = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->getQuery();
    $query->accessCheck(TRUE);
    $query->condition('vid', 'api_products');
    $query->condition('field_menu_link.value', $product_link);
    $term_ids = $query->execute();

    if (empty($term_ids)) {
      return new JsonResponse(['error' => 'No product found for the given link'], 404);
    }

    $term_id = reset($term_ids);
    $term = Term::load($term_id);

    // Load the node which references this taxonomy term in field_product_updates
    $node_query = \Drupal::entityTypeManager()->getStorage('node')->getQuery();
    $node_query->accessCheck(TRUE);
    $node_query->condition('type', 'product_details');
    $node_query->condition('field_product_updates.target_id', $term_id);
    $node_query->range(0, 1);
    $node_ids = $node_query->execute();

    if (empty($node_ids)) {
      return new JsonResponse(['error' => 'No product node found for the given link'], 404);
    }

    $node_id = reset($node_ids);
    $node = Node::load($node_id);

    // Get Product Name with Language Fallback
    $product_name = $this->getTranslatedFieldValue($node, 'field_product_name', $requested_lan);

    // Get Product Menu Link (from taxonomy term) with Language Fallback
    $product_menu_link = $this->getTranslatedFieldValue($term, 'field_menu_link', $requested_lan);

    return new JsonResponse([
      'product_name' => $product_name,
      'product_menu_link' => $product_menu_link,
    ]);
  }
  // public function getProductFaqContent(Request $request)
  // {
  //   try {
  //     $content = json_decode($request->getContent(), true);
  //     $requested_langcode = $content['lan'] ?? '';
  //     $product_link = $content['product_link'] ?? '';

  //     if (empty($requested_langcode) || empty($product_link)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => empty($requested_langcode) ? 'Language code (lan) is required.' : 'Product Link is required.',
  //       ], 400);
  //     }

  //     $database = \Drupal::database();

  //     // Fetch node IDs for valid about_tab_year_content
  //     $fetchNodeIds = function ($langcode, $product_link) use ($database) {
  //       $query = $database->select('node_field_data', 'nfd')
  //         ->fields('nfd', ['nid'])
  //         ->condition('nfd.type', 'product_details_faq')
  //         ->condition('nfd.status', 1)
  //         ->condition('nfd.langcode', $langcode);
  //       // Join node -> taxonomy reference
  //       $query->join('node__field_select_product', 'fsp', 'fsp.entity_id = nfd.nid');

  //       // Join taxonomy term text field table
  //       $query->join('taxonomy_term__field_menu_link', 'tlink', 'tlink.entity_id = fsp.field_select_product_target_id');

  //       // Text fields use *_value column
  //       $query->condition('tlink.field_menu_link_value', $product_link, '=');
  //       return $query->execute()->fetchCol();
  //     };

  //     $nids = $fetchNodeIds($requested_langcode, $product_link);
  //     $fallback = false;

  //     // Fallback to English if no nodes found
  //     if (empty($nids) && $requested_langcode !== 'en') {
  //       $requested_langcode = 'en';
  //       $nids = $fetchNodeIds('en', $product_link);
  //       $fallback = true;
  //     }

  //     if (empty($nids)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'No FAQs found for the given product link.',
  //       ], 404);
  //     }
  //     $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
  //     $response = [];

  //     $language = \Drupal::languageManager()->getLanguage($requested_langcode);
  //     $language_name = $language ? $language->getName() : $requested_langcode;

  //     foreach ($nodes as $node) {
  //       if ($node->hasTranslation($requested_langcode)) {
  //         $node = $node->getTranslation($requested_langcode);
  //       }

  //       $nid = $node->id();

  //       if ($node->hasField('field_api_faqs') && !$node->get('field_api_faqs')->isEmpty()) {
  //         foreach ($node->get('field_api_faqs') as $para_ref) {
  //           $para = $para_ref->entity;
  //           if ($para && $para->hasTranslation($requested_langcode)) {
  //             $para = $para->getTranslation($requested_langcode);
  //           }

  //           $question = $para->get('field_question')->value ?? '';
  //           $answer = $para->get('field_answer')->value ?? '';

  //           $response[] = [
  //             'question' => $question,
  //             'answer' => $answer,
  //           ];
  //         }
  //       }
  //       $response = [
  //         'nid' => $nid,
  //         'langcode' => $requested_langcode,
  //         'language_name' => $language_name,
  //         'faqs' => $response
  //       ];
  //     }
  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
  //       'data' => $response,
  //     ]);
  //   } catch (\Exception $e) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred' . $e->getMessage(),
  //     ], 500);
  //   }
  // }
  public function getProductFaqContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $product_link = $content['product_link'] ?? '';

      if (empty($requested_langcode) || empty($product_link)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($requested_langcode) ? 'Language code (lan) is required.' : 'Product Link is required.',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode, $product_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'product_details_faq')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        // Join node -> taxonomy reference
        $query->join('node__field_select_product', 'fsp', 'fsp.entity_id = nfd.nid');

        // Join taxonomy term text field table
        $query->join('taxonomy_term__field_menu_link', 'tlink', 'tlink.entity_id = fsp.field_select_product_target_id');

        // Text fields use *_value column
        $query->condition('tlink.field_menu_link_value', $product_link, '=');
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $product_link);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $product_link);
        $fallback = true;
      }


      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();

        if ($node->hasField('field_api_faqs') && !$node->get('field_api_faqs')->isEmpty()) {
          foreach ($node->get('field_api_faqs') as $para_ref) {
            $para = $para_ref->entity;
            if ($para && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }

            $question = $para->get('field_question')->value ?? '';
            $answer = $para->get('field_answer')->value ?? '';

            $response[] = [
              'question' => $question,
              'answer' => $answer,
            ];
          }
        }
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'faqs' => $response
        ];
      }

      if (empty($response)) {
        $response = [
          'nid' => '',
          'langcode' => '',
          'language_name' => '',
          'faqs' => ''
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }
}
