<?php

namespace Drupal\api_module\Controller\productCategoryDetails;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;

/**
 * Returns product category details.
 */
class CategoryController extends ControllerBase
{

  /**
   * Returns details for a product category by term ID.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   */
  /* Category List Page */


  public function getBannerCategoryList(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 66);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getCategoryList(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Get requested language or fallback to 'en'
      $requested_langcode = isset($data['lan']) ? $data['lan'] : 'en';

      // Pagination setup
      $page = isset($data['page']) ? max(1, (int)$data['page']) : 1;
      $limit = isset($data['limit']) ? max(1, (int)$data['limit']) : 10;
      $offset = ($page - 1) * $limit;

      // --- Step 1: Try fetching terms in requested language ---
      $query_lang = $database->select('taxonomy_term_field_data', 'td');
      $query_lang->distinct();
      $query_lang->fields('td', ['tid', 'name', 'langcode']);
      $query_lang->leftJoin('taxonomy_term__field_category_link', 'link', 'link.entity_id = td.tid');
      $query_lang->addExpression('MAX(link.field_category_link_value)', 'category_link');
      $query_lang->condition('td.vid', 'therapeutic_category');
      $query_lang->condition('td.status', 1);
      $query_lang->condition('td.langcode', $requested_langcode);
      $query_lang->groupBy('td.tid');
      $query_lang->groupBy('td.name');
      $query_lang->groupBy('td.langcode');
      $query_lang->orderBy('td.name', 'ASC');
      $results_lang = $query_lang->execute()->fetchAll();

      // --- Step 2: Fallback to English if no results in requested language ---
      if (empty($results_lang) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';

        $query_fallback = $database->select('taxonomy_term_field_data', 'td');
        $query_fallback->distinct();
        $query_fallback->fields('td', ['tid', 'name', 'langcode']);
        $query_fallback->leftJoin('taxonomy_term__field_category_link', 'link', 'link.entity_id = td.tid');
        $query_fallback->addExpression('MAX(link.field_category_link_value)', 'category_link');
        $query_fallback->condition('td.vid', 'therapeutic_category');
        $query_fallback->condition('td.status', 1);
        $query_fallback->condition('td.langcode', 'en');
        $query_fallback->groupBy('td.tid');
        $query_fallback->groupBy('td.name');
        $query_fallback->groupBy('td.langcode');
        $query_fallback->orderBy('td.name', 'ASC');
        $results_lang = $query_fallback->execute()->fetchAll();
      }

      // Pagination manually applied on result set
      $total_items = count($results_lang);
      $paged_terms = array_slice($results_lang, $offset, $limit);

      // Get all language names
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();

      $response_data = [];
      foreach ($paged_terms as $term) {
        $langcode = $term->langcode;
        $language_name = isset($lang_names[$langcode]) ? $lang_names[$langcode][0] : ucfirst($langcode);

        $response_data[] = [
          'tid' => (int) $term->tid,
          'name' => (string) $term->name,
          'category_link' => isset($term->category_link) ? (string) $term->category_link : '',
          'langcode' => $langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'Category list fetched successfully.',
        'data' => $response_data,
        'pagination' => [
          'total' => $total_items,
          'page' => $page,
          'limit' => $limit,
          'total_pages' => ceil($total_items / $limit),
        ],
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }



  /*  END */

  /*  START CATEGORY DETAILS SECTION */


  public function getbannerdetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $category_link = $data['category_link'] ?? NULL;

      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      $requested_langcode = $data['lan'] ?? 'en';

      if (!$category_link) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Category parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Function to fetch banner for a language
      $fetchBanner = function ($langcode) use ($database, $category_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        // Join for breadcrumb
        $query->leftJoin('node__field_category_banner_breadcrumb', 'breadcrumb', 'breadcrumb.entity_id = nfd.nid AND breadcrumb.langcode = nfd.langcode AND breadcrumb.delta = 0');
        $query->addField('breadcrumb', 'field_category_banner_breadcrumb_value', 'breadcrumb');

        // Join for image
        $query->leftJoin('node__field_category_banner_image', 'image', 'image.entity_id = nfd.nid AND image.langcode = nfd.langcode AND image.delta = 0');
        $query->addField('image', 'field_category_banner_image_target_id', 'banner_image');

        // Join for brochure
        $query->leftJoin('node__field_category_banner_brochure_l', 'brochure', 'brochure.entity_id = nfd.nid AND brochure.langcode = nfd.langcode AND brochure.delta = 0');
        $query->addField('brochure', 'field_category_banner_brochure_l_value', 'brochure_link');

        // Join for taxonomy reference
        $query->leftJoin('node__field_cat_name', 'catname', 'catname.entity_id = nfd.nid AND catname.delta = 0');

        // Join to match category link
        $query->leftJoin('taxonomy_term__field_category_link', 'term_link', 'term_link.entity_id = catname.field_cat_name_target_id');
        $query->condition('term_link.field_category_link_value', $category_link);

        // Get taxonomy name
        $query->leftJoin('taxonomy_term_field_data', 'term_data', 'term_data.tid = catname.field_cat_name_target_id');
        $query->addField('term_data', 'name', 'category_name');

        $query->condition('nfd.type', 'category_details_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->range(0, 1);
        $query->distinct();

        return $query->execute()->fetchObject();
      };

      // Try in requested language
      $row = $fetchBanner($requested_langcode);

      // Fallback to English if nothing found
      if (empty($row) && $requested_langcode !== 'en') {
        $row = $fetchBanner('en');
        $requested_langcode = 'en';
      }

      if (!$row) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No banner data found for the given category',
          'data' => (object)[],
        ]);
      }

      // Load image URL
      $banner_image_url = '';
      if (!empty($row->banner_image)) {
        $file = \Drupal\file\Entity\File::load($row->banner_image);
        if ($file) {
          $banner_image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }

      // Language name
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$row->langcode][0] ?? ucfirst($row->langcode);

      $response_data = [
        'nid' => $row->nid,
        'title' => $row->title ?? '',
        'banner_navigation' => $row->breadcrumb ?? '',
        'banner_image' => $banner_image_url,
        'banner_download_brochure_link' => $row->brochure_link ?? '',
        'category_name' => $row->category_name ?? '',
        'langcode' => $row->langcode,
        'language_name' => $language_name,
      ];

      return new JsonResponse([
        'status' => true,
        'message' => 'Banner data fetched successfully',
        'data' => $response_data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getCategoryWiseProduct(Request $request)
  {
    try {
      $fallback = false;
      $data = json_decode((string) $request->getContent(), TRUE) ?: [];


      if (empty($data['lan']) || empty($data['category_link'])) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($data['lan']) ? 'Language Parameter is missing' : 'Category parameter is missing',
        ], 400);
      }

      $requested_langcode = $data['lan'] ?? 'en';
      $category_link      = $data['category_link'];
      $page               = isset($data['page'])  ? (int) $data['page']  : 1;
      $limit              = isset($data['limit']) ? (int) $data['limit'] : 10;
      $offset             = ($page - 1) * $limit;

      $entity_repo  = \Drupal::service('entity.repository');
      $lang_manager = \Drupal::languageManager();
      $lang_names   = $lang_manager->getStandardLanguageList();

      $buildQuery = function (string $langcode) use ($category_link, $limit, $offset) {
        return \Drupal::entityQuery('node')
          ->accessCheck(FALSE) // raw SQL didn't do node access checks
          ->condition('type', 'product_details')
          ->condition('status', 1)
          ->condition('langcode', $langcode)
          // Match *first* referenced term only (delta 0), like your SQL join.
          ->condition('field_therapeutic_category.0.entity:taxonomy_term.field_category_link', $category_link) // docs show delta + entity path
          ->range($offset, $limit);
      };

      $buildCount = function (string $langcode) use ($category_link) {
        return \Drupal::entityQuery('node')
          ->accessCheck(FALSE)
          ->condition('type', 'product_details')
          ->condition('status', 1)
          ->condition('langcode', $langcode)
          ->condition('field_therapeutic_category.0.entity:taxonomy_term.field_category_link', $category_link)
          ->count();
      };

      $nids  = $buildQuery($requested_langcode)->execute();
      $total = (int) $buildCount($requested_langcode)->execute();

      $active_langcode = $requested_langcode;
      if (empty($nids) && $requested_langcode !== 'en') {
        $fallback = true;
        $active_langcode = 'en';
        $nids  = $buildQuery('en')->execute();
        $total = (int) $buildCount('en')->execute();
      }

      if (empty($nids) && $total === 0) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No products found for the given category link.',
          'data' => [],
          'pagination' => [
            'total' => 0,
            'page' => $page,
            'limit' => $limit,
            'total_pages' => 0,
          ],
        ], 404);
      }

      $nodes = Node::loadMultiple($nids);
      $response_data = [];

      foreach ($nodes as $node) {
        $node_t = $entity_repo->getTranslationFromContext($node, $active_langcode);
        $product_name = $node_t->get('field_product_name')->value ?? '';
        $product_link = $node_t->get('field_product_link')->value ?? '';
        $term = $node_t->get('field_therapeutic_category')->entity;

        $category_name = '';
        if ($term) {
          $term_t = \Drupal::service('entity.repository')->getTranslationFromContext($term, $active_langcode);
          $category_name = $term_t->label();
        }

        $response_data[] = [
          'nid'           => $node_t->id(),
          'product_name'  => $product_name,
          'product_link'  => $product_link,
          'category_name' => $category_name,
          'langcode'      => $node_t->language()->getId(),
          'language_name' => $lang_names[$node_t->language()->getId()][0] ?? ucfirst($node_t->language()->getId()),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English as no data found in the requested language.' : 'Data fetched Successfully',
        'data' => $response_data,
        'pagination' => [
          'total' => $total,
          'page' => $page,
          'limit' => $limit,
          'total_pages' => (int) ceil($total / $limit),
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  // public function getCategoryWiseProduct(Request $request)
  // {
  //   try {
  //     $content = $request->getContent();
  //     $data = json_decode($content, TRUE);


  //     if (!$data['lan']) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Language Parameter is missing',
  //       ], 400);
  //     }
  //     $category_link = $data['category_link'] ?? NULL;
  //     $page = isset($data['page']) ? (int)$data['page'] : 1;
  //     $limit = isset($data['limit']) ? (int)$data['limit'] : 10;
  //     $requested_langcode = $data['lan'] ?? 'en';

  //     if (!$category_link) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Category parameter is missing',
  //       ], 400);
  //     }

  //     $offset = ($page - 1) * $limit;
  //     $database = \Drupal::database();

  //     // Helper function to get product data
  //     $fetchProducts = function ($langcode) use ($database, $category_link, $offset, $limit) {
  //       $query = $database->select('node_field_data', 'nfd')
  //         ->fields('nfd', ['nid', 'langcode']);

  //       // Join field_product_name with langcode filter
  //       $query->leftJoin('node__field_product_name', 'pn', 'pn.entity_id = nfd.nid AND pn.langcode = nfd.langcode AND pn.delta = 0');
  //       $query->addField('pn', 'field_product_name_value', 'product_name');

  //       // Join field_product_link with langcode filter
  //       $query->leftJoin('node__field_product_link', 'pl', 'pl.entity_id = nfd.nid AND pl.langcode = nfd.langcode AND pl.delta = 0');
  //       $query->addField('pl', 'field_product_link_value', 'product_link');

  //       // Join therapeutic category
  //       $query->leftJoin('node__field_therapeutic_category', 'cat', 'cat.entity_id = nfd.nid AND cat.delta = 0');
  //       $query->leftJoin('taxonomy_term__field_category_link', 'tlink', 'tlink.entity_id = cat.field_therapeutic_category_target_id');
  //       $query->condition('tlink.field_category_link_value', $category_link);

  //       // Get category name
  //       $query->leftJoin('taxonomy_term_field_data', 'td', 'td.tid = cat.field_therapeutic_category_target_id');
  //       $query->addField('td', 'name', 'category_name');

  //       // Apply conditions
  //       $query->condition('nfd.type', 'product_details');
  //       $query->condition('nfd.status', 1);
  //       $query->condition('nfd.langcode', $langcode);fa
  //       $query->distinct();
  //       $query->range($offset, $limit);

  //       return $query->execute()->fetchAll();
  //     };

  //     // Helper for count
  //     $countProducts = function ($langcode) use ($database, $category_link) {
  //       $count_query = $database->select('node_field_data', 'nfd');
  //       $count_query->leftJoin('node__field_therapeutic_category', 'cat', 'cat.entity_id = nfd.nid AND cat.delta = 0');
  //       $count_query->leftJoin('taxonomy_term__field_category_link', 'tlink', 'tlink.entity_id = cat.field_therapeutic_category_target_id');
  //       $count_query->condition('tlink.field_category_link_value', $category_link);
  //       $count_query->condition('nfd.type', 'product_details');
  //       $count_query->condition('nfd.status', 1);
  //       $count_query->condition('nfd.langcode', $langcode);
  //       $count_query->addExpression('COUNT(DISTINCT nfd.nid)', 'total');
  //       return $count_query->execute()->fetchField();
  //     };

  //     // Try requested language first
  //     $results = $fetchProducts($requested_langcode);
  //     $total = $countProducts($requested_langcode);

  //     // Fallback to English if no data
  //     if (empty($results) && $requested_langcode !== 'en') {
  //       $requested_langcode = 'en';
  //       $results = $fetchProducts('en');
  //       $total = $countProducts('en');
  //     }

  //     // Language name helper
  //     $lang_names = \Drupal::languageManager()->getStandardLanguageList();

  //     $response_data = [];
  //     foreach ($results as $row) {
  //       $response_data[] = [
  //         'nid' => $row->nid,
  //         'product_name' => $row->product_name ?? '',
  //         'product_link' => $row->product_link ?? '',
  //         'category_name' => $row->category_name ?? '',
  //         'langcode' => $row->langcode,
  //         'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
  //       ];
  //     }

  //     return new JsonResponse([
  //       'status' => true,
  //       'data' => $response_data,
  //       'pagination' => [
  //         'total' => (int) $total,
  //         'page' => $page,
  //         'limit' => $limit,
  //         'total_pages' => ceil($total / $limit),
  //       ],
  //     ]);
  //   } catch (\Exception $e) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred: ' . $e->getMessage(),
  //     ], 500);
  //   }
  // }


  public function getCategoryAboutContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $category_link = $data['category_link'] ?? NULL;
      $requested_langcode = $data['lan'] ?? 'en';


      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      if (!$category_link) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Category parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Helper function to fetch data by language
      $fetchAboutContent = function ($langcode) use ($database, $category_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_category_about_heading', 'nfcabhead', 'nfcabhead.entity_id = nfd.nid AND nfcabhead.langcode = nfd.langcode AND nfcabhead.delta = 0');
        $query->addField('nfcabhead', 'field_category_about_heading_value', 'heading');

        $query->leftJoin('node__field_category_about_section', 'nfcades', 'nfcades.entity_id = nfd.nid AND nfcades.langcode = nfd.langcode AND nfcades.delta = 0');
        $query->addField('nfcades', 'field_category_about_section_value', 'description');

        $query->leftJoin('node__field_categ_name', 'catname', 'catname.entity_id = nfd.nid AND catname.delta = 0');
        $query->leftJoin('taxonomy_term__field_category_link', 'term_link', 'term_link.entity_id = catname.field_categ_name_target_id');
        $query->condition('term_link.field_category_link_value', $category_link);

        $query->leftJoin('taxonomy_term_field_data', 'term_data', 'term_data.tid = catname.field_categ_name_target_id');
        $query->addField('term_data', 'name', 'category_name');

        $query->condition('nfd.type', 'category_details_about_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->range(0, 1);
        $query->distinct();

        return $query->execute()->fetchObject();
      };

      // Try requested language
      $row = $fetchAboutContent($requested_langcode);

      // Fallback to English if needed
      if (empty($row) && $requested_langcode !== 'en') {
        $row = $fetchAboutContent('en');
        $requested_langcode = 'en';
      }

      if (!$row) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No content found for the given category',
          'data' => (object)[],
        ]);
      }

      // Language name
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$row->langcode][0] ?? ucfirst($row->langcode);

      $response_data = [
        'nid' => $row->nid,
        'heading' => $row->heading ?? '',
        'description' => $row->description ?? '',
        'category_name' => $row->category_name ?? '',
        'langcode' => $row->langcode,
        'language_name' => $language_name,
      ];

      return new JsonResponse([
        'status' => true,
        'message' => 'Category about content fetched successfully',
        'data' => $response_data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getCategoryDescriptionContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $category_link = $data['category_link'] ?? NULL;
      $requested_langcode = $data['lan'] ?? 'en';


      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      if (!$category_link) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Category parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Helper function to fetch description content by language
      $fetchDescriptionContent = function ($langcode) use ($database, $category_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_category_description_text', 'nfcades', 'nfcades.entity_id = nfd.nid AND nfcades.langcode = nfd.langcode AND nfcades.delta = 0');
        $query->addField('nfcades', 'field_category_description_text_value', 'description');

        $query->leftJoin('node__field_ca_name', 'catname', 'catname.entity_id = nfd.nid AND catname.delta = 0');

        $query->leftJoin('taxonomy_term__field_category_link', 'term_link', 'term_link.entity_id = catname.field_ca_name_target_id');
        $query->condition('term_link.field_category_link_value', $category_link);

        $query->leftJoin('taxonomy_term_field_data', 'term_data', 'term_data.tid = catname.field_ca_name_target_id');
        $query->addField('term_data', 'name', 'category_name');

        $query->condition('nfd.type', 'category_details_description_sec');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->range(0, 1);
        $query->distinct();

        return $query->execute()->fetchObject();
      };

      // Try requested language
      $row = $fetchDescriptionContent($requested_langcode);

      // Fallback to English if nothing found
      if (empty($row) && $requested_langcode !== 'en') {
        $row = $fetchDescriptionContent('en');
        $requested_langcode = 'en';
      }

      if (!$row) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No content found for the given category',
          'data' => (object)[],
        ]);
      }

      // Get readable language name
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$row->langcode][0] ?? ucfirst($row->langcode);

      $response_data = [
        'nid' => $row->nid,
        'description' => $row->description ?? '',
        'category_name' => $row->category_name ?? '',
        'langcode' => $row->langcode,
        'language_name' => $language_name,
      ];

      return new JsonResponse([
        'status' => true,
        'message' => 'Category description content fetched successfully',
        'data' => $response_data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getCategoryUSPsContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $category_link = $data['category_link'] ?? NULL;
      $requested_langcode = $data['lan'] ?? 'en';


      if (!$data['lan']) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      if (!$category_link) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Category parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Helper function to fetch USP data by language
      $fetchUSPContent = function ($langcode) use ($database, $category_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_category_usp_heading', 'nfcahead', 'nfcahead.entity_id = nfd.nid AND nfcahead.langcode = nfd.langcode AND nfcahead.delta = 0');
        $query->addField('nfcahead', 'field_category_usp_heading_value', 'heading');

        $query->leftJoin('node__field_category_usp_text', 'nfcades', 'nfcades.entity_id = nfd.nid AND nfcades.langcode = nfd.langcode AND nfcades.delta = 0');
        $query->addField('nfcades', 'field_category_usp_text_value', 'description');

        $query->leftJoin('node__field_category_usp_img', 'nfcaimg', 'nfcaimg.entity_id = nfd.nid AND nfcaimg.langcode = nfd.langcode AND nfcaimg.delta = 0');
        $query->addField('nfcaimg', 'field_category_usp_img_target_id', 'image');

        $query->leftJoin('node__field_cate_name', 'catname', 'catname.entity_id = nfd.nid AND catname.delta = 0');
        $query->leftJoin('taxonomy_term__field_category_link', 'term_link', 'term_link.entity_id = catname.field_cate_name_target_id');
        $query->condition('term_link.field_category_link_value', $category_link);

        $query->leftJoin('taxonomy_term_field_data', 'term_data', 'term_data.tid = catname.field_cate_name_target_id');
        $query->addField('term_data', 'name', 'category_name');

        $query->condition('nfd.type', 'category_details_usp_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->range(0, 1);
        $query->distinct();

        return $query->execute()->fetchObject();
      };

      // Try requested language
      $row = $fetchUSPContent($requested_langcode);

      // Fallback to English if nothing found
      if (empty($row) && $requested_langcode !== 'en') {
        $row = $fetchUSPContent('en');
        $requested_langcode = 'en';
      }

      if (!$row) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No content found for the given category',
          'data' => (object)[],
        ]);
      }

      // Generate image URL
      $image_url = '';
      if (!empty($row->image)) {
        $file = \Drupal\file\Entity\File::load($row->image);
        if ($file) {
          $image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }

      // Get readable language name
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $language_name = $lang_names[$row->langcode][0] ?? ucfirst($row->langcode);

      $response_data = [
        'nid' => $row->nid,
        'heading' => $row->heading ?? '',
        'description' => $row->description ?? '',
        'image' => $image_url,
        'category_name' => $row->category_name ?? '',
        'langcode' => $row->langcode,
        'language_name' => $language_name,
      ];

      return new JsonResponse([
        'status' => true,
        'message' => 'Category USPs content fetched successfully',
        'data' => $response_data,
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getCategoryFaqContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $category_link = $content['category_link'] ?? '';

      if (empty($requested_langcode) || empty($category_link)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($requested_langcode) ? 'Language parameter is missing' : 'Category parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs by category link
      $fetchNodeIds = function ($langcode) use ($database, $category_link) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'category_details_faq')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->leftJoin('node__field_c_name', 'catname', 'catname.entity_id = nfd.nid AND catname.langcode = nfd.langcode');
        $query->leftJoin('taxonomy_term__field_category_link', 'term_link', 'term_link.entity_id = catname.field_c_name_target_id');
        $query->condition('term_link.field_category_link_value', $category_link);

        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      if (empty($nids)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No FAQs found for the given category',
          'data' => [],
        ]);
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $title = $node->getTitle();
        $category_name = '';

        // Extract taxonomy category name
        if ($node->hasField('field_c_name') && !$node->get('field_c_name')->isEmpty()) {
          $term = $node->get('field_c_name')->entity;
          if ($term && $term->hasTranslation($requested_langcode)) {
            $term = $term->getTranslation($requested_langcode);
          }
          $category_name = $term ? $term->getName() : '';
        }


        if ($node->hasField('field_category_faq') && !$node->get('field_category_faq')->isEmpty()) {
          foreach ($node->get('field_category_faq') as $faq_ref) {
            $para = $faq_ref->entity;
            if ($para && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }
            $question = $para->get('field_question')->value ?? '';
            $answer = $para->get('field_answer')->value ?? '';

            $response[] = [
              'nid' => $nid,
              'langcode' => $requested_langcode,
              'language_name' => $language_name,
              'title' => $title,
              'question' => $question,
              'answer' =>  $answer,
              'category_name' => $category_name
            ];
          }
        }
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'FAQs fetched successfully.',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  // public function getCategoryFaqContent(Request $request)
  // {
  //   try {
  //     $content = $request->getContent();
  //     $data = json_decode($content, TRUE);

  //     $category_link = $data['category_link'] ?? NULL;
  //     $requested_langcode = $data['lan'] ?? 'en';

  //     if (!$data['lan']) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Language Parameter is missing',
  //       ], 400);
  //     }

  //     if (!$category_link) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Category parameter is missing',
  //       ], 400);
  //     }

  //     $database = \Drupal::database();

  //     $fetchFaqContent = function ($langcode) use ($database, $category_link) {
  //       $query = $database->select('node_field_data', 'nfd')
  //         ->fields('nfd', ['nid', 'title', 'langcode']);

  //       // Join node -> taxonomy term via field_c_name
  //       $query->leftJoin('node__field_c_name', 'catname', 'catname.entity_id = nfd.nid AND catname.langcode = nfd.langcode');
  //       $query->leftJoin('taxonomy_term__field_category_link', 'term_link', 'term_link.entity_id = catname.field_c_name_target_id');
  //       $query->condition('term_link.field_category_link_value', $category_link);

  //       // Get taxonomy term name
  //       $query->leftJoin('taxonomy_term_field_data', 'term_data', 'term_data.tid = catname.field_c_name_target_id');
  //       $query->addField('term_data', 'name', 'category_name');

  //       // Paragraph reference from node
  //       $query->leftJoin('node__field_category_faq', 'faq_ref', 'faq_ref.entity_id = nfd.nid AND faq_ref.langcode = nfd.langcode');
  //       $query->addField('faq_ref', 'field_category_faq_target_id', 'paragraph_id');

  //       // Paragraph base table
  //       $query->leftJoin('paragraphs_item_field_data', 'pfd', 'pfd.id = faq_ref.field_category_faq_target_id AND pfd.langcode = nfd.langcode');
  //       $query->condition('pfd.status', 1);
  //       $query->condition('pfd.type', 'faq');

  //       // Question and Answer
  //       $query->leftJoin('paragraph__field_question', 'pq', 'pq.entity_id = pfd.id AND pq.langcode = nfd.langcode');
  //       $query->leftJoin('paragraph__field_answer', 'pa', 'pa.entity_id = pfd.id AND pa.langcode = nfd.langcode');
  //       $query->addField('pq', 'field_question_value', 'question');
  //       $query->addField('pa', 'field_answer_value', 'answer');

  //       $query->condition('nfd.type', 'category_details_faq');
  //       $query->condition('nfd.status', 1);
  //       $query->condition('nfd.langcode', $langcode);
  //       $query->distinct();

  //       return $query->execute()->fetchAll();
  //     };

  //     // Try requested language first
  //     $results = $fetchFaqContent($requested_langcode);

  //     // Fallback to English if not found
  //     if (empty($results) && $requested_langcode !== 'en') {
  //       $results = $fetchFaqContent('en');
  //       $requested_langcode = 'en';
  //     }

  //     if (empty($results)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'No FAQs found for the given category',
  //         'data' => [],
  //       ]);
  //     }

  //     // Language name
  //     $lang_names = \Drupal::languageManager()->getStandardLanguageList();
  //     $language_name = $lang_names[$requested_langcode][0] ?? ucfirst($requested_langcode);

  //     $response_data = [];
  //     foreach ($results as $row) {
  //       $response_data[] = [
  //         'nid' => (int) $row->nid,
  //         'title' => (string) $row->title,
  //         'category_name' => (string) $row->category_name,
  //         'question' => $row->question ?? '',
  //         'answer' => $row->answer ?? '',
  //         'langcode' => $row->langcode,
  //         'language_name' => $language_name,
  //       ];
  //     }

  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => 'FAQs fetched successfully.',
  //       'data' => $response_data,
  //     ]);
  //   } catch (\Throwable $th) {
  //     \Drupal::logger('custom_module')->error($th->getMessage());
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred: ' . $th->getMessage(),
  //     ], 500);
  //   }
  // }





  /*  END */
}
