<?php

namespace Drupal\api_module\Controller\about;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class AboutController.
 *
 * Provides a controller for the About page.
 */
class AboutController extends ControllerBase
{

  /**
   * Returns a renderable array for the About page.
   *
   * @return array
   *   A simple renderable array.
   */


  public function faqcontent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (!$langcode) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      // Prepare query
      $query = \Drupal::database()->select('node_field_data', 'nfd');

      // Join directly with required fields
      $query->join('node__field_question', 'q', 'q.entity_id = nfd.nid AND q.langcode = :langcode AND q.delta = 0');
      $query->join('node__field_faq_answer', 'a', 'a.entity_id = nfd.nid AND a.langcode = :langcode AND a.delta = 0');

      // Fetch only needed fields
      $query->addField('nfd', 'nid');
      $query->addField('q', 'field_question_value', 'question');
      $query->addField('a', 'field_faq_answer_value', 'answer');

      // WHERE conditions
      $query->condition('nfd.type', 'about_us_faq_section');
      $query->condition('nfd.status', 1);
      $query->condition('nfd.langcode', $langcode);

      $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

      $result = $query->execute();

      // Collect results
      $data = [];
      $language_name = \Drupal::languageManager()->getLanguage($langcode)?->getName() ?? $langcode;

      foreach ($result as $row) {
        $data[] = [
          'nid' => $row->nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'question' => $row->question,
          'answer' => $row->answer,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'FAQ fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function aboutContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Try fetching in requested language first
      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid'])
        ->condition('nfd.type', 'about_us_content')
        ->condition('nfd.status', 1)
        ->range(0, 1);

      $query->leftJoin('node__field_about_us_content', 'c', 'c.entity_id = nfd.nid AND c.langcode = :langcode AND c.delta = 0');
      $query->addField('c', 'field_about_us_content_value', 'content');

      $query->leftJoin('node__field_about_us_heading', 'h', 'h.entity_id = nfd.nid AND h.langcode = :langcode AND h.delta = 0');
      $query->addField('h', 'field_about_us_heading_value', 'heading');

      $query->addExpression(':langcode', 'langcode_used', [':langcode' => $requested_langcode]);

      $row = $query->execute()->fetch();

      $used_langcode = $requested_langcode;

      // If either content or heading is missing, fallback to English
      if (empty($row) || (empty($row->content) && empty($row->heading))) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'about_us_content')
          ->condition('nfd.status', 1)
          ->range(0, 1);

        $query->leftJoin('node__field_about_us_content', 'c', 'c.entity_id = nfd.nid AND c.langcode = :langcode AND c.delta = 0');
        $query->addField('c', 'field_about_us_content_value', 'content');

        $query->leftJoin('node__field_about_us_heading', 'h', 'h.entity_id = nfd.nid AND h.langcode = :langcode AND h.delta = 0');
        $query->addField('h', 'field_about_us_heading_value', 'heading');

        $query->addExpression(':langcode', 'langcode_used', [':langcode' => 'en']);

        $row = $query->execute()->fetch();
        $used_langcode = 'en';
      }

      if (empty($row)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No about content found for the requested or fallback language.',
        ], 404);
      }

      $language_name = \Drupal::languageManager()->getLanguage($used_langcode)?->getName() ?? $used_langcode;

      $data = [
        'nid' => $row->nid,
        'langcode' => $used_langcode,
        'language_name' => $language_name,
        'about_heading' => $row->heading ?? '',
        'about_content' => $row->content ?? '',
      ];

      return new JsonResponse([
        'status' => true,
        'message' => 'About Content data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }





  public function aboutCapabilities(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (!$langcode) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid'])
        ->condition('nfd.type', 'about_us_our_capabilities_sectio')
        ->condition('nfd.status', 1)
        ->condition('nfd.langcode', $langcode);

      $query->leftJoin('node__field_about_us_our_cap_card_desc', 'desc', 'desc.entity_id = nfd.nid AND desc.langcode = nfd.langcode AND desc.delta = 0');
      $query->addField('desc', 'field_about_us_our_cap_card_desc_value', 'description');

      $query->leftJoin('node__field_about_cap_card_heading', 'head', 'head.entity_id = nfd.nid AND head.langcode = nfd.langcode AND head.delta = 0');
      $query->addField('head', 'field_about_cap_card_heading_value', 'heading');

      $query->leftJoin('node__field_about_cap_card_icon', 'icon', 'icon.entity_id = nfd.nid AND icon.langcode = nfd.langcode AND icon.delta = 0');
      $query->addField('icon', 'field_about_cap_card_icon_target_id', 'icon');

      $query->leftJoin('node__field_about_cap_card_image', 'img', 'img.entity_id = nfd.nid AND img.langcode = nfd.langcode AND img.delta = 0');
      $query->addField('img', 'field_about_cap_card_image_target_id', 'image');

      $query->leftJoin('node__field_about_cap_card_link', 'link', 'link.entity_id = nfd.nid AND link.langcode = nfd.langcode AND link.delta = 0');
      $query->addField('link', 'field_about_cap_card_link_value', 'link');

      $results = $query->execute();

      $data = [];
      $language_name = \Drupal::languageManager()->getLanguage($langcode)?->getName() ?? $langcode;
      $file_url_generator = \Drupal::service('file_url_generator');

      foreach ($results as $row) {
        $image_url = '';
        $icon_url = '';

        if (!empty($row->image)) {
          if ($file = \Drupal\file\Entity\File::load($row->image)) {
            $image_url = $file_url_generator->generateAbsoluteString($file->getFileUri());
          }
        }

        if (!empty($row->icon)) {
          if ($file = \Drupal\file\Entity\File::load($row->icon)) {
            $icon_url = $file_url_generator->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $row->nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'description' => $row->description ?? '',
          'heading' => $row->heading ?? '',
          'contentImg_url' => $image_url,
          'contentIcon' => $icon_url,
          'link' => $row->link ?? '',
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'About capabilities data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function aboutSocialResponsibility(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (!$langcode) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();
      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid'])
        ->condition('nfd.type', 'about_us_social_responsibility')
        ->condition('nfd.status', 1)
        ->condition('nfd.langcode', $langcode);

      $query->leftJoin('node__field_about_social_resp_img', 'img', 'img.entity_id = nfd.nid AND img.langcode = nfd.langcode AND img.delta = 0');
      $query->addField('img', 'field_about_social_resp_img_target_id', 'social_resp_img');

      $query->leftJoin('node__field_about_social_resp_desc', 'desc', 'desc.entity_id = nfd.nid AND desc.langcode = nfd.langcode AND desc.delta = 0');
      $query->addField('desc', 'field_about_social_resp_desc_value', 'social_resp_value');

      $result = $query->execute();

      $data = [];
      $language_name = \Drupal::languageManager()->getLanguage($langcode)?->getName() ?? $langcode;
      $file_url_generator = \Drupal::service('file_url_generator');

      foreach ($result as $row) {
        $image_url = '';

        if (!empty($row->social_resp_img)) {
          if ($file = \Drupal\file\Entity\File::load($row->social_resp_img)) {
            $image_url = $file_url_generator->generateAbsoluteString($file->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'description' => $row->social_resp_value ?? '',
          'social_resp_img' => $image_url,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'About Social Responsibility data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function aboutDisclaimer(Request $request)
  {
    try {
      // Decode JSON input and get language code
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid'])
        ->condition('nfd.type', 'about_us_desclaimer')
        ->condition('nfd.status', 1)
        ->condition('nfd.langcode', $langcode)
        ->distinct();

      $query->leftJoin('node__field_about_us_disclaimer_text', 'nfaudt', 'nfaudt.entity_id = nfd.nid AND nfaudt.langcode = nfd.langcode AND nfaudt.delta = 0');
      $query->addField('nfaudt', 'field_about_us_disclaimer_text_value', 'disclaimer_text');

      $result = $query->execute();

      $data = [];
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;
      foreach ($result as $row) {
        $data = [
          'langcode' => $langcode,
          'language_name' => $language_name,
          'nid' => $row->nid,
          'disclaimer_text' => $row->disclaimer_text ?? '',
        ];
      }

      // Get language name from langcode


      return new JsonResponse([
        'status' => true,
        'message' => 'About Disclaimer fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function aboutTabYearContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'about_tab_year_content')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $year_group = $node->get('field_about_tab_group_year')->value ?? '';
        

        $response[$nid] = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'year_group' => $year_group,
          'title' => "",
          'year_contents' => "",
        ];

        // Process paragraph items
        if ($node->hasField('field_about_year_group_content') && !$node->get('field_about_year_group_content')->isEmpty()) {
          foreach ($node->get('field_about_year_group_content') as $para_ref) {
            $para = $para_ref->entity;
            if ($para && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }

            $heading = $para->get('field_about_tab_year_number')->value ?? '';
            $content_val = $para->get('field_about_tab_year_based_conte')->value ?? '';
            $response[$nid]['title'] = $heading;
            $response[$nid]['year_contents'] = $content_val;
          }
        }
      }

      $response = array_values($response);

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to english as no data found in the requested language' : 'About Content data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getbannerdetails(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $nids = \Drupal::entityQuery('node')
        ->condition('type', 'about_us_banner_section')
        ->condition('status', 1)
        ->accessCheck(TRUE)
        ->execute();

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      $data = [];

      foreach ($nodes as $node) {
        // Check if translation exists
        if ($node->hasTranslation($langcode)) {
          $node = $node->getTranslation($langcode);
        }

        $nid = $node->id();
        $title = $node->label();
        $banner_heading = $node->get('field_about_banner_heading')->value ?? '';

        // Get image URL
        $bannerImg_url = '';
        if (!$node->get('field_about_banner')->isEmpty()) {
          $image = $node->get('field_about_banner')->entity;
          if ($image) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($image->getFileUri());
          }
        }
        $brochure = '';
        if (!$node->get('field_about_banner_brochure')->isEmpty()) {
          $brochureFile = $node->get('field_about_banner_brochure')->entity;
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }
        $data[] = [
          'nid' => $nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'banner_navigation' => $title,
          'banner_heading' => $banner_heading,
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  public function getLeadershipData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'about_us_leadership')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $order = $node->get('field_order')->value ?? '';
        $name = $node->get('field_person_s_name')->value ?? '';
        $designation = $node->get('field_designation')->value ?? '';
        $bio = $node->get('field_bio_details')->value ?? '';
        $link = $node->get('field_profile_link')->value ?? '';
        $image = $node->get('field_profile_image')->target_id ?? '';

        $imageURL = "";
        if (!empty($image)) {
          $file = \Drupal\file\Entity\File::load($image);
          if ($file) {
            $imageURL = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        $response[] = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'order' => $order,
          'name' => $name,
          'image' => $imageURL ?? '',
          'designation' => $designation,
          'bio' => $bio,
          'link' => $link
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }
}
