<?php

namespace Drupal\api_module\Controller;

use Exception;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

class UpdatesController extends ControllerBase {

  public function getProductUpdates(Request $request) {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? 'en';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter (lan) is required.',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs by langcode
      $fetchNodeIds = function ($langcode) use ($database) {
        return $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'api_products_updates')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode)
          ->execute()
          ->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      if (empty($nids) && $requested_langcode !== 'en') {
        $nids = $fetchNodeIds('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        // Get referenced entities
        $year_term = $node->get('field_update_year')->entity;
        $month_term = $node->get('field_update_month')->entity;
        $product_term = $node->get('field_select_api_product')->entity;
        $filing_term = $node->get('field_regulatory_filling_detail')->entity;

        $year = $year_term ? $year_term->label() : 'Unknown Year';
        $month = $month_term ? $month_term->label() : 'Unknown Month';
        $filing = $filing_term ? $filing_term->label() : '';
        $product_menu_link = $product_term && $product_term->hasField('field_menu_link')
          ? $product_term->get('field_menu_link')->value
          : '';
        $product_tid = $product_term ? $product_term->id() : null;
        $product_name_text = $node->get('field_api_product_name')->value ?? '';
        $product_name = $product_term ? $product_term->label() : $product_name_text;

        if (!isset($response[$year])) {
          $response[$year] = [];
        }

        if (!isset($response[$year][$month])) {
          $response[$year][$month] = [];
        }

        $response[$year][$month][] = [
          'product_id' => $product_tid,
          // 'product_name' => $product_term ? strip_tags(trim($product_term->label())) : strip_tags(trim($product_name_text)),
          'product_name' => $product_name_text,
          'regulatory_filing_detail' => $filing,
          'menu_link' => $product_menu_link,
        ];
      }

      // Sort years descending
      krsort($response);

      // Sort months descending inside each year using month order
      $month_order = [
        'January' => 1, 'February' => 2, 'March' => 3,
        'April' => 4, 'May' => 5, 'June' => 6,
        'July' => 7, 'August' => 8, 'September' => 9,
        'October' => 10, 'November' => 11, 'December' => 12,
      ];

      foreach ($response as &$months) {
        uksort($months, function ($a, $b) use ($month_order) {
          return ($month_order[$b] ?? 0) <=> ($month_order[$a] ?? 0);
        });
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English content.' : 'Product update data fetched successfully.',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }



}
?>
