<?php

namespace Drupal\api_module\Controller;

use Exception;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;
use Drupal\file\Entity\File;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\taxonomy\Entity\Term;
use Drupal\Core\Language\LanguageInterface;


class TestController extends ControllerBase
{
  public function getTestData(Request $request)
  {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      $product_link = $data['product_link'] ?? '';
      $requested_langcode = $data['lan'] ?? '';

      if (empty($product_link) || empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($product_link) ? 'Product Link parameter is missing' : 'language parameter is missing',
        ], 400);
      }

      //loading the product node
      $nids = \Drupal::entityTypeManager()
        ->getStorage('node')
        ->getQuery()
        ->condition('type', 'product_details')
        ->condition('field_product_link', $product_link)
        ->condition('status', 1)
        ->accessCheck(FALSE)
        ->execute();

      if (empty($nids)) {
        return new JsonResponse([
          'status'  => TRUE,
          'message' => 'No matching product found',
          'data'    => [],
        ], 200);
      }

      $node = Node::load(reset($nids));   // there is only one
      $extractMenuLinks = function (array $term_ids) {
        $links = [];
        foreach ($term_ids as $delta) {
          $term = \Drupal::entityTypeManager()
            ->getStorage('taxonomy_term')
            ->load($delta['target_id']);

          if (!$term || !$term->hasField('field_menu_link')) {
            continue;
          }
          if ($term->hasTranslation('en')) {
            $term = $term->getTranslation('en');
          }
          if (!$term->get('field_menu_link')->isEmpty()) {
            $links[] = $term->get('field_menu_link')->value;
          }
        }
        return $links;
      };
      $related_links = [
        'articles'    => $node->hasField('field_related_articles')
          ? $extractMenuLinks($node->get('field_related_articles')->getValue())
          : [],
        'news'        => $node->hasField('field_related_news')
          ? $extractMenuLinks($node->get('field_related_news')->getValue())
          : [],
        'white-paper' => $node->hasField('field_related_white_papers')
          ? $extractMenuLinks($node->get('field_related_white_papers')->getValue())
          : [],
      ];

      $response_data = [
        'nid'                   => $node->id(),
        'title'                 => $node->getTitle(),
        'product_link'          => $node->get('field_product_link')->value,
        'related_articles'      => $this->ProductResHelper($requested_langcode, $related_links['articles'], 'articles'),
        'related_white_papers'  => $this->ProductResHelper($requested_langcode, $related_links['white-paper'], 'white-paper'),
        'related_news'          => $this->ProductResHelper($requested_langcode, $related_links['news'], 'news'),
        'related_events'        => [],
        'related_webinars'      => [],
      ];

      return new JsonResponse([
        'status'  => true,
        'message' => 'Product data retrieved successfully',
        'langcode' => $requested_langcode ?? 'en',
        'data'    => $response_data,
      ], 200);
    } catch (\Throwable $th) {
      \Drupal::logger('product_api')->error($th->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }
  function ProductResHelper($LANGUAGE, $HARD_LINK, $RESOURCE_TYPE)
  {
    // $LANGUAGE = 'en';
    // $HARD_CODED_LINK = ['asia-pacific-climate-leader-award'];

    $nids = \Drupal::entityQuery('node')
      ->condition('type', 'resources_details')
      ->accessCheck(FALSE)
      ->execute();

    $nodes = Node::loadMultiple($nids);
    $resources = [];

    $path_alias_manager = \Drupal::service('path_alias.manager');
    $entity_repo        = \Drupal::service('entity.repository');

    foreach ($nodes as $node) {
      $node = $entity_repo->getTranslationFromContext($node, $LANGUAGE, ['fallback' => TRUE]);

      if ($node->language()->getId() === LanguageInterface::LANGCODE_NOT_SPECIFIED) {
        continue;
      }

      $type_term = $node->get('field_res_type')->entity;
      if (!$type_term) {
        continue;
      }

      $type_term = $entity_repo->getTranslationFromContext($type_term, $LANGUAGE, ['fallback' => TRUE]);
      if (strtolower($type_term->label()) !== $RESOURCE_TYPE) {
        continue;
      }

      foreach ($node->get('field_resource_details') as $ref) {
        $year_para = $ref->entity;
        if (
          !$year_para
          || $year_para->getEntityTypeId() !== 'paragraph'
          || $year_para->bundle() !== 'resource_year_wise'
        ) {
          continue;
        }

        $year_para = $entity_repo->getTranslationFromContext($year_para, $LANGUAGE, ['fallback' => TRUE]);

        foreach ($year_para->get('field_res_details') as $detail_ref) {
          $detail_para = $detail_ref->entity;
          if (!$detail_para || $detail_para->bundle() !== 'resource_details') {
            continue;
          }

          $detail_para = $entity_repo->getTranslationFromContext($detail_para, $LANGUAGE, ['fallback' => TRUE]);
          $resource_url   = '';
          $resource_term  = $detail_para->get('field_resource_name')->entity;
          if ($resource_term && $resource_term->hasField('field_menu_link') && !$resource_term->get('field_menu_link')->isEmpty()) {
            $term_trans = $entity_repo->getTranslationFromContext($resource_term, $LANGUAGE, ['fallback' => TRUE]);
            $term_en    = $entity_repo->getTranslationFromContext($resource_term, 'en', ['fallback' => TRUE]);

            $raw_path_trans = $term_trans->get('field_menu_link')->value;
            $raw_path_en    = $term_en->get('field_menu_link')->value;

            $normalize = function ($raw) {
              if (empty($raw)) {
                return '';
              }
              if (strpos($raw, '/') !== FALSE) {
                return ltrim($raw, '/');
              }
              if (preg_match('/^\D*?(\d+)$/', $raw, $m)) {
                return 'taxonomy/term/' . $m[1];
              }
              return ltrim($raw, '/');
            };

            $internal_en    = $normalize($raw_path_en);
            $internal_trans = $normalize($raw_path_trans);

            $english_alias_raw    = $path_alias_manager->getAliasByPath('/' . $internal_en, 'en');
            $translated_alias_raw = $path_alias_manager->getAliasByPath('/' . $internal_trans, $LANGUAGE);

            $english_path    = is_string($english_alias_raw) ? ltrim(parse_url($english_alias_raw, PHP_URL_PATH), '/') : '';
            $translated_path = is_string($translated_alias_raw) ? ltrim(parse_url($translated_alias_raw, PHP_URL_PATH), '/') : '';

            if (!in_array($english_path, $HARD_LINK, TRUE)) {
              continue;
            }

            $final_alias = (!empty($translated_path) && $translated_path !== $english_path)
              ? $translated_alias_raw
              : $english_alias_raw;

            $resource_url = $final_alias;
          }
          // ------------------------------------

          $title       = $detail_para->get('field_resource_main_heading')->value ?? '';
          $description = $detail_para->hasField('field_res_description')
            && !$detail_para->get('field_res_description')->isEmpty()
            ? strip_tags($detail_para->get('field_res_description')->value)
            : '';

          $image_url = '';

          if (
            $detail_para->hasField('field_resource_feat_image') &&
            !$detail_para->get('field_resource_feat_image')->isEmpty()
          ) {
            // Use featured image first if available
            $file = $detail_para->get('field_resource_feat_image')->entity;
            if ($file) {
              $image_url = \Drupal::service('file_url_generator')
                ->generateAbsoluteString($file->getFileUri());
            }
          } elseif (
            $detail_para->hasField('field_resource_main_image') &&
            !$detail_para->get('field_resource_main_image')->isEmpty()
          ) {
            // Fallback to main image if featured not found
            $file = $detail_para->get('field_resource_main_image')->entity;
            if ($file) {
              $image_url = \Drupal::service('file_url_generator')
                ->generateAbsoluteString($file->getFileUri());
            }
          }
          $url = $resource_url;
          if (str_starts_with($url, '/')) {
            $url = ltrim($url, '/');
          }
          $resources[] = [
            'title'       => $title,
            'description' => $description,
            'url'         => $url,
            'image'       => $image_url,
          ];
        }
      }
    }

    return $resources;
  }


  // public function getTestData(Request $request)
  // {
  //   try {
  //     $content = json_decode($request->getContent(), true);

  //     if (!isset($content['lan']) || !isset($content['content_type'])) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Both "lan" and "content_type" are required.'
  //       ], 400);
  //     }

  //     $langcode = $content['lan'];
  //     [$content_type, $is_multi_node] = $content['content_type'];
  //     $original_langcode = $langcode;

  //     $normal_fields = $content['normal_field'] ?? [];
  //     $file_fields = $content['file/image'] ?? [];
  //     $paragraph_fields = $content['paragraph'] ?? [];
  //     $taxonomy_fields = $content['taxonomy'] ?? [];
  //     $conditions = $content['condition'] ?? [];

  //     // Query building
  //     $query = \Drupal::entityQuery('node')
  //       ->accessCheck(FALSE)
  //       ->condition('type', $content_type)
  //       ->condition('status', 1)
  //       ->condition('langcode', $langcode);

  //     foreach ($conditions as $condition) {
  //       [$value, $field_name, $is_taxonomy] = $condition;
  //       $query->condition($field_name, $value);
  //     }

  //     $nids = $query->execute();

  //     $fallback = false;
  //     if (empty($nids) && $langcode !== 'en') {
  //       $fallback = true;
  //       $langcode = 'en';
  //       $query = \Drupal::entityQuery('node')
  //         ->accessCheck(FALSE)
  //         ->condition('type', $content_type)
  //         ->condition('status', 1)
  //         ->condition('langcode', $langcode);

  //       foreach ($conditions as $condition) {
  //         [$value, $field_name, $is_taxonomy] = $condition;
  //         $query->condition($field_name, $value);
  //       }

  //       $nids = $query->execute();
  //     }

  //     if ($is_multi_node === 'no' && !empty($nids)) {
  //       $nids = [reset($nids)];
  //     }

  //     $nodes = Node::loadMultiple($nids);
  //     $results = [];

  //     foreach ($nodes as $node) {
  //       if ($node->hasTranslation($langcode)) {
  //         $node = $node->getTranslation($langcode);
  //       }

  //       $data = [
  //         'nid' => $node->id(),
  //         'langcode' => $langcode
  //       ];

  //       // Normal fields
  //       foreach ($normal_fields as $field_name => $is_multi) {
  //         if ($node->hasField($field_name)) {
  //           $values = $node->get($field_name)->getValue();
  //           $data[$field_name] = ($is_multi === 'yes') ? array_column($values, 'value') : ($values[0]['value'] ?? '');
  //         }
  //       }

  //       // File/Image fields
  //       foreach ($file_fields as $field_name => $is_multi) {
  //         if ($node->hasField($field_name)) {
  //           $items = $node->get($field_name)->getValue();
  //           $urls = [];
  //           foreach ($items as $item) {
  //             $fid = $item['target_id'] ?? null;
  //             if ($fid) {
  //               $file = File::load($fid);
  //               if ($file) {
  //                 $urls[] = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
  //               }
  //             }
  //           }
  //           $data[$field_name] = ($is_multi === 'yes') ? $urls : ($urls[0] ?? '');
  //         }
  //       }

  //       // Paragraph fields
  //       foreach ($paragraph_fields as $field_name => $target_type) {
  //         if ($node->hasField($field_name)) {
  //           $field = $node->get($field_name);
  //           if ($field && $field instanceof \Drupal\Core\Field\EntityReferenceFieldItemListInterface) {
  //             $refs = $field->referencedEntities();
  //           }
  //           $resolved = [];
  //           foreach ($refs as $ref) {
  //             if ($ref instanceof TranslatableInterface && $ref->hasTranslation($langcode)) {
  //               $ref = $ref->getTranslation($langcode);
  //             }
  //             $resolved[] = $this->extractEntityFields($ref, $langcode);
  //           }
  //           $data[$field_name] = $resolved;
  //         }
  //       }

  //       // Taxonomy fields
  //       foreach ($taxonomy_fields as $field_name => $target_type) {
  //         if ($node->hasField($field_name)) {
  //           $field = $node->get($field_name);
  //           if ($field && $field instanceof \Drupal\Core\Field\EntityReferenceFieldItemListInterface) {
  //             $refs = $field->referencedEntities();
  //           }
  //           $resolved = [];
  //           foreach ($refs as $ref) {
  //             if ($ref instanceof TranslatableInterface && $ref->hasTranslation($langcode)) {
  //               $ref = $ref->getTranslation($langcode);
  //             }
  //             $resolved[] = $this->extractEntityFields($ref, $langcode);
  //           }
  //           $data[$field_name] = $resolved;
  //         }
  //       }

  //       $results[] = $data;
  //     }

  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => $fallback ? 'Fallback to English as no data was found in the requested language.' : 'Data fetched successfully.',
  //       'data' => $results
  //     ]);
  //   } catch (\Exception $e) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'Error: ' . $e->getMessage()
  //     ], 500);
  //   }
  // }
  private static function extractEntityFields($entity, $langcode)
  {
    $output = ['id' => $entity->id()];

    foreach ($entity->getFields() as $field_name => $field) {
      if (strpos($field_name, 'field_') !== 0 || $field->isEmpty()) {
        continue;
      }

      $value = $field->getValue();

      if (isset($value[0]['target_id']) && $field->getFieldDefinition()->getType() === 'image') {
        $urls = [];
        foreach ($value as $val) {
          $file = File::load($val['target_id']);
          if ($file) {
            $urls[] = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        $output[$field_name] = count($urls) > 1 ? $urls : ($urls[0] ?? '');
      } elseif (isset($value[0]['value'])) {
        $output[$field_name] = count($value) > 1 ? array_column($value, 'value') : $value[0]['value'];
      }
    }

    return $output;
  }
}
