<?php

namespace Drupal\api_module\Controller;

use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Controller\ControllerBase;
use Exception;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;

/**
 * Class SearchController
 *
 * Provides a controller to handle search requests.
 */
class SearchController extends ControllerBase
{
  /**
   * Handles the search request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing the search query.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the search results.
   */

  /*  public function globalSearchList(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? 'en';  // fallback to 'en'
      $search = trim($content['search'] ?? '');
      $limit = $content['limit'] ?? 10;
      $page = max(1, (int) ($content['page'] ?? 1));
      $offset = ($page - 1) * $limit;
      $searchWildcard = '%' . str_replace(' ', '%', $search) . '%';

      $database = \Drupal::database();
      $data = ['categories' => [], 'products' => [], 'pagination' => []];

      // 1. Fetch matching categories
      $term_query = $database->select('taxonomy_term_field_data', 'ttfd');
      $term_query->leftJoin('taxonomy_term__field_category_link', 'tml', 'tml.entity_id = ttfd.tid AND tml.deleted = 0');
      $term_query->fields('ttfd', ['tid', 'name']);
      $term_query->addField('tml', 'field_category_link_value');
      $term_query->condition('ttfd.vid', 'therapeutic_category');
      $term_query->condition('ttfd.langcode', $langcode);
      $term_query->condition('ttfd.name', $searchWildcard, 'LIKE');
      $terms = $term_query->execute()->fetchAllAssoc('tid');

      // 2. Fetch matching products (paginated)
      $product_query = $database->select('node_field_data', 'n');
      $product_query->leftJoin('node__field_product_name', 'fpn', 'fpn.entity_id = n.nid AND fpn.deleted = 0');
      $product_query->leftJoin('node__field_product_link', 'fpl', 'fpl.entity_id = n.nid AND fpl.deleted = 0');
      $product_query->leftJoin('node__field_product_about_text', 'fpabt', 'fpabt.entity_id = n.nid AND fpabt.deleted = 0');
      $product_query->leftJoin('node__field_therapeutic_category', 'ftc', 'ftc.entity_id = n.nid AND ftc.deleted = 0');
      $product_query->leftJoin('taxonomy_term_field_data', 'tc_name', 'tc_name.tid = ftc.field_therapeutic_category_target_id AND tc_name.langcode = :langcode', [':langcode' => $langcode]);
      $product_query->leftJoin('node__field_dr_reddy_s_development_sta', 'fds', 'fds.entity_id = n.nid AND fds.deleted = 0');
      $product_query->leftJoin('taxonomy_term_field_data', 'drds', 'drds.tid = fds.field_dr_reddy_s_development_sta_target_id AND drds.langcode = :langcode', [':langcode' => $langcode]);

      $product_query->fields('n', ['nid']);
      $product_query->addField('fpn', 'field_product_name_value', 'product_name');
      $product_query->addField('fpl', 'field_product_link_value', 'product_link');
      $product_query->addField('fpabt', 'field_product_about_text_value', 'product_about_text');
      $product_query->addField('drds', 'name', 'development_status');
      $product_query->addField('tc_name', 'name', 'therapeutic_category');

      $group = $product_query->orConditionGroup()
        ->condition('n.title', $searchWildcard, 'LIKE')
        ->condition('fpn.field_product_name_value', $searchWildcard, 'LIKE')
        ->condition('tc_name.name', $searchWildcard, 'LIKE');

      $product_query->condition('n.status', 1);
      $product_query->condition('n.langcode', $langcode);
      $product_query->condition('n.type', 'product_details');
      $product_query->condition($group);
      $product_query->range($offset, $limit);
      $product_query->orderBy('n.created', 'DESC');

      $products = $product_query->execute()->fetchAllAssoc('nid');

      // 3. Count total matching products (fixed)
      $count_query = $database->select('node_field_data', 'n');
      $count_query->leftJoin('node__field_product_name', 'fpn', 'fpn.entity_id = n.nid AND fpn.deleted = 0');
      $count_query->leftJoin('node__field_therapeutic_category', 'ftc', 'ftc.entity_id = n.nid AND ftc.deleted = 0');
      $count_query->leftJoin('taxonomy_term_field_data', 'tc_name', 'tc_name.tid = ftc.field_therapeutic_category_target_id AND tc_name.langcode = :langcode', [':langcode' => $langcode]);

      $count_query->condition('n.status', 1);
      $count_query->condition('n.langcode', $langcode);
      $count_query->condition('n.type', 'product_details');
      $count_query->condition(
        $count_query->orConditionGroup()
          ->condition('n.title', $searchWildcard, 'LIKE')
          ->condition('fpn.field_product_name_value', $searchWildcard, 'LIKE')
          ->condition('tc_name.name', $searchWildcard, 'LIKE')
      );

      // Fix: Use COUNT(DISTINCT n.nid)
      $count_query->addExpression('COUNT(DISTINCT n.nid)', 'total_count');
      $total = $count_query->execute()->fetchField();

      // 4. Format categories
      foreach ($terms as $term) {
        $data['categories'][] = [
          'tid' => $term->tid,
          'name' => $term->name,
          'field_menu_link_value' => $term->field_category_link_value ?? '',
        ];
      }

      // 5. Format products
      foreach ($products as $node) {
        $data['products'][] = [
          'nid' => $node->nid,
          'field_product_name_value' => $node->product_name ?? '',
          'field_product_link_value' => $node->product_link ?? '',
          'field_product_about_text_value' => isset($node->product_about_text)
            ? implode(' ', array_slice(explode(' ', strip_tags($node->product_about_text)), 0, 50))
            : '',
          'dr_reddy_development_status' => $node->development_status ?? '',
          'therapeutic_category_name' => $node->therapeutic_category ?? '',
        ];
      }

      // 6. Pagination info
      $data['pagination'] = [
        'total' => (int) $total,
        'page' => $page,
        'limit' => $limit,
      ];

      // 7. Final response
      if (empty($data['categories']) && empty($data['products'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No records found.',
        ], 200);
      }

      return new JsonResponse([
        'status' => true,
        'data' => $data,
      ], 200);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'error' => $e->getMessage(),
      ], 500);
    }
  } */

  public function globalSearchList(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $search_string = trim($content['search'] ?? '');
      $limit = (int) ($content['limit'] ?? 10);
      $page = max((int) ($content['page'] ?? 1), 1);
      $offset = ($page - 1) * $limit;

      if (empty($requested_langcode) || empty($search_string)) {
        return new JsonResponse([
          'status' => false,
          'message' => !empty($search_string) ? 'Language code (lan) is required.' : 'Please specify a keyword to search.',
        ], 400);
      }

      $fetchNodeIds = function ($requested_langcode, $search_string, $offset, $limit, $page) {
        $top_products = [];
        $matched_categories = [];
        $category_only_products = [];
        $total_category_matched_count = 0;

        // 1️⃣ Exact match products (unpaginated)
        $exact_match_nids = \Drupal::entityQuery('node')
          ->condition('type', 'product_details')
          ->condition('status', 1)
          ->condition('langcode', $requested_langcode)
          ->condition('field_product_name', '%' . $search_string . '%', 'LIKE')
          ->accessCheck(TRUE);

        $orGroup = $exact_match_nids->orConditionGroup()
          ->condition('field_hide_frontend', '0')
          ->notExists('field_hide_frontend');
        $exact_match_nids->condition($orGroup);

        $exact_match_nids =   $exact_match_nids->execute();

        $top_products = Node::loadMultiple($exact_match_nids);
        $top_count = count($top_products);

        // 2️⃣ Taxonomy term match
        $matched_term_ids = \Drupal::entityQuery('taxonomy_term')
          ->condition('vid', 'therapeutic_category')
          ->condition('langcode', $requested_langcode)
          ->condition('name', '%' . $search_string . '%', 'LIKE')
          ->accessCheck(TRUE)
          ->execute();

        if (!empty($matched_term_ids)) {
          $terms = Term::loadMultiple($matched_term_ids);
          foreach ($terms as $term) {
            // Load translated version if available
            $translated_term = $term->hasTranslation($requested_langcode)
              ? $term->getTranslation($requested_langcode)
              : $term;

            $matched_categories[] = [
              'tid' => $translated_term->id(),
              'name' => $translated_term->label(),
              'field_menu_link_value' => $translated_term->get('field_category_link')->value ?? '',
            ];
          }

          // Count total category-matched nodes (for pagination info)
          $total_category_matched_count = \Drupal::entityQuery('node')
            ->condition('type', 'product_details')
            ->condition('status', 1)
            ->condition('langcode', $requested_langcode)
            ->condition('field_therapeutic_category', $matched_term_ids, 'IN')
            ->accessCheck(TRUE)
            ->count()
            ->execute();

          // Load only remaining slot count
          $remaining = max($limit - $top_count, 0);
          if ($remaining > 0) {
            $category_query = \Drupal::entityQuery('node')
              ->condition('type', 'product_details')
              ->condition('status', 1)
              ->condition('langcode', $requested_langcode)
              ->condition('field_therapeutic_category', $matched_term_ids, 'IN')
              ->accessCheck(TRUE)
              ->range(max(0, $offset - $top_count), $remaining);

            $orGroup2 = $category_query->orConditionGroup()
              ->condition('field_hide_frontend', '0', '=')
              ->notExists('field_hide_frontend');
            $category_query->condition($orGroup2);
            
            $category_nids = $category_query->execute();
            $category_only_products = Node::loadMultiple($category_nids);
          }
        }

        // Final merge
        $merged_products = array_merge($top_products, $category_only_products);

        // Format
        $formatted_products = [];
        foreach ($merged_products as $node) {
          // 🌐 Translate node
          $translated_node = $node->hasTranslation($requested_langcode)
            ? $node->getTranslation($requested_langcode)
            : $node;

          // 🌐 Translate category term
          $category_term = $translated_node->get('field_therapeutic_category')->referencedEntities()[0] ?? null;
          $category_term = $category_term && $category_term->hasTranslation($requested_langcode)
            ? $category_term->getTranslation($requested_langcode)
            : $category_term;

          // 🌐 Translate status term
          $status_term = $translated_node->get('field_dr_reddy_s_development_sta')->referencedEntities()[0] ?? null;
          $status_term = $status_term && $status_term->hasTranslation($requested_langcode)
            ? $status_term->getTranslation($requested_langcode)
            : $status_term;

          $formatted_products[] = [
            'nid' => $translated_node->id(),
            'field_product_name_value' => $translated_node->get('field_product_name')->value ?? '',
            'field_product_link_value' => $translated_node->get('field_product_link')->value ?? '',
            'field_product_about_text_value' =>  implode(' ', array_slice(explode(' ', strip_tags($translated_node->get('field_product_about_text')->value)), 0, 50)) ?? '',
            'dr_reddy_development_status' => $status_term?->label() ?? '',
            'therapeutic_category_name' => $category_term?->label() ?? '',
          ];
        }

        // Response structure
        return [
          'categories' => $matched_categories,
          'products' => $formatted_products,
          'pagination' => [
            'total' => count($exact_match_nids) + $total_category_matched_count,
            'page' => $page,
            'limit' => $limit,
          ],
        ];
      };

      $data = $fetchNodeIds($requested_langcode, $search_string, $offset, $limit, $page);

      return new JsonResponse([
        'status' => true,
        'data' => $data,
      ], 200);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'error' => $e->getMessage(),
      ], 500);
    }
  }


  public function searchBanner(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 303);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
}
