<?php

namespace Drupal\api_module\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Returns responses for Meet With Expert routes.
 */
class MeetWithExpertController extends ControllerBase
{

  /**
   * Returns a simple page.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   */
  public function getMeetWithExpertBanner(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = $input['lan'] ?? 'en';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_mae_heading', 'nfbh', "nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = '" . $langcode . "'");
        $query->addField('nfbh', 'field_mae_heading_value', 'banner_heading');

        $query->leftJoin('node__field_mae_banner_cut_out_image', 'nfabi', "nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = '" . $langcode . "'");
        $query->addField('nfabi', 'field_mae_banner_cut_out_image_target_id', 'banner_image');

        $query->leftJoin('node__field_mae_banner_subheading', 'nfbsh', "nfbsh.entity_id = nfd.nid AND nfbsh.delta = 0 AND nfbsh.langcode = '" . $langcode . "'");
        $query->addField('nfbsh', 'field_mae_banner_subheading_value', 'banner_sub_heading');

        $query->condition('nfd.type', 'meet_api_expert_main_section'); // Replace with actual content type
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $results = $fetchData('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];
      $bannerImg_url = '';

      if (!empty($results[0]->banner_image)) {
        $file = \Drupal\file\Entity\File::load($results[0]->banner_image);
        if ($file) {
          $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }

      $data = [
        'nid' => $results[0]->nid,
        'banner_heading' => $results[0]->banner_heading ?? '',
        'banner_sub_heading' => $results[0]->banner_sub_heading ?? '',
        'banner_image' => $bannerImg_url,
        'langcode' => $results[0]->langcode,
        'language_name' => $lang_names[$results[0]->langcode][0] ?? ucfirst($results[0]->langcode),
      ];

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English. No data found in requested language.' : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getMeetWithContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = $input['lan'] ?? 'en';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_mae_section2_desc', 'nfbc', "nfbc.entity_id = nfd.nid AND nfbc.delta = 0 AND nfbc.langcode = '" . $langcode . "'");
        $query->addField('nfbc', 'field_mae_section2_desc_value', 'content');

        $query->leftJoin('node__field_mae_section2_image', 'nfabi', "nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = '" . $langcode . "'");
        $query->addField('nfabi', 'field_mae_section2_image_target_id', 'content_image');

        $query->condition('nfd.type', 'meet_api_expert_section_2'); // Replace with actual content type
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $results = $fetchData('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];
      $content_image_url = '';

      if (!empty($results[0]->content_image)) {
        $file = \Drupal\file\Entity\File::load($results[0]->content_image);
        if ($file) {
          $content_image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }

      $data = [
        'nid' => $results[0]->nid,
        'content' => $results[0]->content ?? '',
        'content_image' => $content_image_url,
        'langcode' => $results[0]->langcode,
        'language_name' => $lang_names[$results[0]->langcode][0] ?? ucfirst($results[0]->langcode),
      ];

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English. No data found in requested language.' : 'Content after banner section fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getMeetWithExpertAPIBusiness(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'meet_api_expert_api_business')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $text = $node->get('field_api_business_text')->value ?? "";
        $icon_url = $node->get('field_api_business_icon')->target_id ?? "";

        $icon = "";
        if (!empty($icon_url)) {
          $file = \Drupal\file\Entity\File::load($icon_url);
          if ($file) {
            $icon = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $response[] = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'text' => $text,
          'icon' => $icon,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }

  public function getMaeVideoData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'meet_api_expert_section_3')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $desc = $node->get('field_mae_sec3_desc')->value ?? '';
        $iframe_link = $node->get('field_iframe_link')->value ?? '';


        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'description' => $desc,
          'iframe_link' => $iframe_link
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  public function getMaeInfoData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'meet_api_expert_section_4')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $desc = $node->get('field_mae_sec4_desc')->value ?? '';
        $image_id = $node->get('field_mae_sec4_image')->target_id ?? '';

        $image = "";
        if (!empty($image_id)) {
          $file1 = \Drupal\file\Entity\File::load($image_id);
          if ($file1) {
            $image = \Drupal::service('file_url_generator')->generateAbsoluteString($file1->getFileUri());
          }
        }

        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'description' => $desc,
          'image' => $image
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
}
