<?php

namespace Drupal\api_module\Controller;

use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Controller\ControllerBase;
use Exception;
use Symfony\Component\HttpFoundation\Request;

class HomeContentController extends ControllerBase
{

  /**
   * Returns a JSON response with home content.
   */

  public function getWhoweareContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';


      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      // Load the latest published node of type 'home_who_we_are_content'
      $nids = \Drupal::entityQuery('node')
        ->condition('type', 'home_who_we_are_content')
        ->condition('status', 1)
        ->sort('created', 'DESC')
        ->range(0, 1)
        ->accessCheck(TRUE)
        ->execute();

      if (empty($nids)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No content found',
          'data' => [],
        ]);
      }

      $nid = reset($nids);
      $node = \Drupal\node\Entity\Node::load($nid);

      if (!$node) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Node not found',
          'data' => [],
        ]);
      }

      // Handle translation
      if ($node->hasTranslation($langcode)) {
        $node = $node->getTranslation($langcode);
      } elseif ($langcode !== $node->language()->getId()) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Translation not available for language code: ' . $langcode,
          'data' => [],
        ]);
      }

      // Get the translated field value
      $home_who_content = $node->get('field_who_we_are_content')->value ?? '';

      $data = [
        'nid' => $nid,
        'langcode' => $langcode,
        'language_name' => $language_name,
        'home_who_content' => $home_who_content,
      ];

      return new JsonResponse([
        'status' => true,
        'message' => 'Data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'Error fetching data: ' . $e->getMessage(),
        'data' => [],
      ], 500);
    }
  }

  public function getWhoWeareNumberContent(Request $request)
  {
    try {
      // Decode JSON payload to get language
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid'])
        ->condition('nfd.type', 'home_who_we_are_ext')
        ->condition('nfd.status', 1)
        ->condition('nfd.langcode', $langcode)
        ->distinct();

      $query->leftJoin('node__field_counter_label', 'nfq', 'nfq.entity_id = nfd.nid AND nfq.langcode = nfd.langcode AND nfq.delta = 0');
      $query->addField('nfq', 'field_counter_label_value', 'counter_level');

      $query->leftJoin('node__field_counter', 'nfa', 'nfa.entity_id = nfd.nid AND nfa.langcode = nfd.langcode AND nfa.delta = 0');
      $query->addField('nfa', 'field_counter_value', 'counter_val');

      $result = $query->execute();

      $data = [];
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;
      foreach ($result as $row) {
        $data[] = [
          'nid' => $row->nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'counter_lebel' => $row->counter_level,
          'counter_value' => $row->counter_val,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'Home who we are counter value fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getXeccedContentData(Request $request)
  {
    try {
      // Decode JSON payload to get language
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $query = $database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid'])
        ->condition('nfd.type', 'home_xceed_section')
        ->condition('nfd.status', 1)
        ->condition('nfd.langcode', $langcode)
        ->distinct();

      $query->leftJoin('node__field_labels', 'nfq', 'nfq.entity_id = nfd.nid AND nfq.langcode = nfd.langcode AND nfq.delta = 0');
      $query->addField('nfq', 'field_labels_value', 'xecced_value');
      $query->leftJoin('node__field_icons', 'nfi', 'nfi.entity_id = nfd.nid AND nfi.langcode = nfd.langcode AND nfi.delta = 0');
      $query->addField('nfi', 'field_icons_target_id', 'icon_url');  // Correct now

      $result = $query->execute();

      $data = [];
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      foreach ($result as $row) {
        $icon_url = "";
        if (!empty($row->icon_url)) {
          $icon_file = \Drupal\file\Entity\File::load($row->icon_url);
          if ($icon_file) {
            $icon_url = \Drupal::service('file_url_generator')->generateAbsoluteString($icon_file->getFileUri());
          }
        }
        $data[] = [
          'nid' => $row->nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'icon' => $icon_url,
          'xecced_value' => $row->xecced_value,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'Xecced value fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getOurStrengthContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      // Get language name
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      // Load all published 'home_our_strength_content' nodes
      $nids = \Drupal::entityQuery('node')
        ->condition('type', 'home_our_strength_content')
        ->condition('status', 1)
        ->accessCheck(TRUE)
        ->execute();

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $data = [];

      foreach ($nodes as $node) {
        // Translate node if translation exists
        if ($node->hasTranslation($langcode)) {
          $node = $node->getTranslation($langcode);
        }

        $nid = $node->id();

        // Strength Title
        $title = $node->get('field_strength_title')->value ?? '';

        // Strength Content
        $content_value = $node->get('field_strength_content')->value ?? '';

        // Icon
        $icon_url = '';
        if (!$node->get('field_strength_icon')->isEmpty()) {
          $file = $node->get('field_strength_icon')->entity;
          if ($file) {
            $icon_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'strength_title' => $title,
          'icon' => $icon_url,
          'detailed_description' => $content_value,
        ];
      }

      return new JsonResponse([
        'data' => $data,
        'status' => true,
        'message' => 'Data fetched successfully',
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'data' => [],
        'status' => false,
        'message' => 'Error: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getAwardContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      // Get language name
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      // Load all published award_section nodes
      $nids = \Drupal::entityQuery('node')
        ->condition('type', 'award_section')
        ->condition('status', 1)
        ->accessCheck(TRUE)
        ->execute();

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $data = [];

      foreach ($nodes as $node) {
        // Translate node if available
        if ($node->hasTranslation($langcode)) {
          $node = $node->getTranslation($langcode);
        }

        $nid = $node->id();

        // Award image
        $award_image_url = '';
        if (!$node->get('field_award_image')->isEmpty()) {
          $image = $node->get('field_award_image')->entity;
          if ($image) {
            $award_image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($image->getFileUri());
          }
        }

        $data[] = [
          'nid' => $nid,
          'langcode' => $langcode,
          'language_name' => $language_name,
          'award_image' => $award_image_url,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => 'Award Section data fetched successfully.',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
}
