<?php

namespace Drupal\api_module\Controller;

use Exception;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;
use Drupal\file\Entity\File;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\paragraphs\Entity\Paragraph;

class GlobalContentController extends ControllerBase
{


  public function getCustomPagesContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $content_page_id = $content['page_id'] ?? '';
      $requested_langcode = $content['lan'] ?? '';
      $banner = [];
      $data = [];


      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      if (empty($content_page_id)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'page_id is required',
        ], 400);
      }

      $database = \Drupal::database();
      $fallback = false;

      // Helper function to fetch a page node
      $fetchPage = function ($langcode, $types) use ($database, $content_page_id) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode'])
          ->condition('nfd.type', $types, 'IN')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode)
          ->distinct();

        // Page type join
        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.langcode = nfd.langcode AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');
        $query->condition('nfptyp.field_page_type_target_id', $content_page_id);

        // Term name (page_type name)
        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        // Content field
        $query->leftJoin('node__field_description', 'nfdesc', 'nfdesc.entity_id = nfd.nid AND nfdesc.langcode = nfd.langcode AND nfdesc.delta = 0');
        $query->addField('nfdesc', 'field_description_value', 'content');

        // Banner fields
        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        return $query->execute()->fetchObject();
      };

      // Step 1: Try to fetch both custom content and banner content in requested language
      $contentRow = $fetchPage($requested_langcode, ['custom_page']);
      $bannerRow = $fetchPage($requested_langcode, ['common_inner_banner_section']);

      // Step 2: Fallback to English if content not found
      if (!$contentRow && $requested_langcode !== 'en') {
        $contentRow = $fetchPage('en', ['custom_page']);
        $bannerRow = $fetchPage('en', ['common_inner_banner_section']);
        $requested_langcode = 'en';
        $fallback = true;
      }

      // Step 3: If still not found, return empty
      if (!$contentRow) {
        return new JsonResponse([
          'status' => true,
          'data' => [],
          'message' => 'No custom page content found',
        ], 404);
      }

      $nid = $contentRow->nid;
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      // Load banner image file
      $bannerImg_url = '';
      if (!empty($bannerRow->banner_image)) {
        $file = \Drupal\file\Entity\File::load($bannerRow->banner_image);
        if ($file) {
          $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }
      }


      // Final response
      $data = [
        'nid' => $nid,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
        'page_type' => $contentRow->page_type_name ?? '',
        'content' => $contentRow->content ?? '',

      ];

      $banner = [
        'banner_heading' => $bannerRow->banner_heading ?? '',
        'banner_navigation' => $bannerRow->banner_navigation_link ?? '',
        'banner_image' => $bannerImg_url,
      ];

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Custom page content fetched successfully',
        'data' => $data,
        'banner' => $banner,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getCustomSectionsContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $content_page_id = $content['page_id'] ?? '';
      $requested_langcode = $content['lan'] ?? '';
      $requested_section = $content['section_id'] ?? '';

      if (empty($requested_langcode) || empty($content_page_id) || empty($requested_section)) {
        $msg = "";
        if (empty($requested_langcode)) $msg = "Language code (lan) is required.";
        if (empty($content_page_id)) $msg = "page_id is required.";
        if (empty($requested_section)) $msg = "section_id is required.";
        return new JsonResponse([
          'status' => false,
          'message' => $msg,
        ], 400);
      }

      $database = \Drupal::database();
      $fallback = false;

      $fetchNodeIds = function ($langcode, $page_type_id, $section_id) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'custom_page')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        //This should be like this, but as the taxonomy terms are not being translated for page_type and sections, we are trying to fetch it in default lang.
        // $query->leftJoin('node__field_page_type', 'nfpt', 'nfpt.entity_id = nfd.nid AND nfpt.langcode = nfd.langcode');
        // $query->condition('nfpt.field_page_type_target_id', $page_type_id);

        // $query->leftJoin('node__field_section_order', 'nfso', 'nfso.entity_id = nfd.nid AND nfso.langcode = nfd.langcode');
        // $query->condition('nfso.field_section_order_target_id', $section_id);

        $query->leftJoin('node__field_page_type', 'nfpt', 'nfpt.entity_id = nfd.nid');
        $query->condition('nfpt.field_page_type_target_id', $page_type_id);

        $query->leftJoin('node__field_section_order', 'nfso', 'nfso.entity_id = nfd.nid');
        $query->condition('nfso.field_section_order_target_id', $section_id);

        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $content_page_id, $requested_section);

      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $content_page_id, $requested_section);
        $fallback = true;
      }

      $nodes = Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node instanceof TranslatableInterface && $node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $title = $node->getTitle();
        $heading = $node->get('field_heading')->value ?? '';
        $subheading = $node->get('field_sub_heading')->value ?? '';
        $description = $node->get('field_description')->value ?? '';
        $reference_link = $node->get('field_reference_link')->value ?? '';

        $common_image = '';
        $common_image_id = $node->get('field_common_image')->target_id ?? '';
        if (!empty($common_image_id)) {
          $file1 = File::load($common_image_id);
          if ($file1) {
            $common_image = \Drupal::service('file_url_generator')->generateAbsoluteString($file1->getFileUri());
          }
        }

        $innerDatas = [];
        if ($node->hasField('field_custom_sections') && !$node->get('field_custom_sections')->isEmpty()) {
          foreach ($node->get('field_custom_sections') as $key => $para_ref) {
            $para = $para_ref->entity;
            if ($para && $para instanceof TranslatableInterface && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }

            $id = $para->id() ?? '';
            $_heading = $para->get('field_heading')->value ?? '';
            $_subheading = $para->get('field_sub_heading')->value ?? '';
            $_description = $para->get('field_description')->value ?? '';
            $_link = $para->get('field_link')->value ?? '';
            $image_id = $para->get('field_image')->target_id ?? '';

            $image = '';
            if (!empty($image_id)) {
              $file = File::load($image_id);
              if ($file) {
                $image = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
              }
            }

            $innerDatas[$key] = [
              'content_id' => $id,
              'heading' => $_heading,
              'sub_heading' => $_subheading,
              'description' => $_description,
              'link' => $_link,
              'image' => $image,
            ];
          }
        }

        $response = [
          "lang_id" => $requested_langcode,
          "language_name" => $language_name,
          "nid" => $nid,
          "title" => $title,
          "heading" => $heading,
          "sub_heading" => $subheading,
          "description" => $description,
          "image" => $common_image,
          "link" => $reference_link,
          "contents" => $innerDatas,
        ];
      }

      if (!$response) {
        return new JsonResponse([
          'status' => true,
          'data' => [],
          'message' => 'No custom page content found',
        ], 404);
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Data fetched successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getHpApiesBannerDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 65);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function EventGalleryPageWise(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';
      $page_type_id = $input['page_flag_id'];

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      if (empty($page_type_id)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Page type flag is missing.',
        ], 400);
      }

      $database = \Drupal::database();
      $fetchData = function ($langcode, $page_type_id) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        // First image only (delta = 0)
        $query->leftJoin(
          'node__field_image_elements',
          'nfabi',
          'nfabi.entity_id = nfd.nid AND nfabi.langcode = :langcode'
        );
        $query->addField('nfabi', 'field_image_elements_target_id', 'gallery_image');

        // Join node reference field to taxonomy
        $query->leftJoin(
          'node__field_select_plantaion_drive',
          'nfptyp',
          'nfptyp.entity_id = nfd.nid AND nfptyp.langcode = :langcode'
        );

        // Join taxonomy term to get name
        $query->leftJoin(
          'taxonomy_term_field_data',
          'ttfd',
          'ttfd.tid = nfptyp.field_select_plantaion_drive_target_id AND ttfd.langcode = :langcode'
        );
        $query->addField('ttfd', 'name', 'page_type_name');

        // Conditions
        $query->condition('nfd.type', 'plantation_drive_gallery');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_select_plantaion_drive_target_id', $page_type_id);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        // Grouping to avoid duplicates
        $query->groupBy('nfd.nid');
        $query->groupBy('nfd.title');
        $query->groupBy('nfd.langcode');
        $query->groupBy('nfabi.field_image_elements_target_id');
        $query->groupBy('ttfd.name');

        return $query->execute()->fetchAll();
      };



      $results = $fetchData($requested_langcode, $page_type_id);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en', $page_type_id);
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $gallery_image_url = '';
        if (!empty($row->gallery_image)) {
          $file = \Drupal\file\Entity\File::load($row->gallery_image);
          if ($file) {
            $gallery_image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $row->nid,
          'banner_image' => $gallery_image_url,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getMainSectionData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'hpapis_main_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $top_text = $node->get('field_hpapis_main_section_top_te')->value ?? '';
        $bottom_text = $node->get('field_hpapis_main_section_bottom')->value ?? '';
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'top_text' => $top_text,
          'bottom_text' => $bottom_text,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getHpApisFacilitySectionData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'hpapis_facilities_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $heading = $node->get('field_hpapis_facilities_heading')->value ?? '';
        $text = $node->get('field_hpapis_facilities_text')->value ?? '';
        $main_image_id = $node->get('field_hpapis_facilities_main_ima')->target_id ?? '';
        $below_image_id = $node->get('field_hpapis_facilities_below_im')->target_id ?? '';


        $main_image = "";
        $below_image = "";
        if (!empty($main_image_id)) {
          $file1 = \Drupal\file\Entity\File::load($main_image_id);
          if ($file1) {
            $main_image = \Drupal::service('file_url_generator')->generateAbsoluteString($file1->getFileUri());
          }
        }
        if (!empty($below_image_id)) {
          $file2 = \Drupal\file\Entity\File::load($below_image_id);
          if ($file2) {
            $below_image = \Drupal::service('file_url_generator')->generateAbsoluteString($file2->getFileUri());
          }
        }
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'heading' => $heading,
          'text' => $text,
          'main_image' => $main_image,
          'below_image' => $below_image,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getHpApisFaqData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'hpapis_faq')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();

        if ($node->hasField('field_hpapis_faq') && !$node->get('field_hpapis_faq')->isEmpty()) {
          foreach ($node->get('field_hpapis_faq') as $para_ref) {
            $para = $para_ref->entity;
            if ($para && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }

            $question = $para->get('field_question')->value ?? '';
            $answer = $para->get('field_answer')->value ?? '';

            $response[] = [
              'question' => $question,
              'answer' => $answer,
            ];
          }
        }
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'faqs' => $response,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }

  public function getHpApisGraffitiData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'hpapis_graffiti_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $heading = $node->get('field_hpapis_graffiti_heading')->value ?? '';
        if ($node->hasField('field_hpapi_graffitti') && !$node->get('field_hpapi_graffitti')->isEmpty()) {
          foreach ($node->get('field_hpapi_graffitti') as $para_ref) {
            $para = $para_ref->entity;
            if ($para && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }

            $icon_id = $para->get('field_hpapis_graffiti_icon')->target_id ?? '';
            $text = $para->get('field_hpapis_graffiti_text')->value ?? '';

            $icon = "";
            if (!empty($icon_id)) {
              $file = \Drupal\file\Entity\File::load($icon_id);
              if ($file) {
                $icon = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
              }
            }
            $response[] = [
              'icon' => $icon,
              'text' => $text,
            ];
          }
        }
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'heading' => $heading,
          'faqs' => $response,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }

  public function TressListPlanationDriveInnerPage(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = $input['lan'] ?? 'en';
      $page_type = $input['page_type'] ?? '';
      $page = (int)($input['page'] ?? 1);
      $limit = (int)($input['limit'] ?? 10);
      $offset = ($page - 1) * $limit;

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      if (empty($page_type)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Page Type parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Get total count
      $total_query = $database->select('node_field_data', 'nfd');
      $total_query->leftJoin('node__field_select_drive', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
      $total_query->condition('nfd.type', 'plantation_drive_people');
      $total_query->condition('nfd.status', 1);
      $total_query->condition('nfptyp.field_select_drive_target_id', $page_type);
      $total_query->condition('nfd.langcode', $requested_langcode);
      $total = $total_query->countQuery()->execute()->fetchField();

      $fetchData = function ($langcode, $page_type, $limit, $offset) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_name', 'nfna', 'nfna.entity_id = nfd.nid AND nfna.delta = 0 AND nfna.langcode = :langcode');
        $query->addField('nfna', 'field_name_value', 'treesname');

        $query->leftJoin('node__field_trees_count', 'nftressc', 'nftressc.entity_id = nfd.nid AND nftressc.delta = 0 AND nftressc.langcode = :langcode');
        $query->addField('nftressc', 'field_trees_count_value', 'trees_count');

        $query->leftJoin('node__field_select_drive', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_select_drive_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_select_drive_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'plantation_drive_people');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_select_drive_target_id', $page_type);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->range($offset, $limit);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode, $page_type, $limit, $offset);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en', $page_type, $limit, $offset);
        $fallback = true;

        // Recalculate total for fallback language
        $total_query = $database->select('node_field_data', 'nfd');
        $total_query->leftJoin('node__field_select_drive', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $total_query->condition('nfd.type', 'plantation_drive_people');
        $total_query->condition('nfd.status', 1);
        $total_query->condition('nfptyp.field_select_drive_target_id', $page_type);
        $total_query->condition('nfd.langcode', 'en');
        $total = $total_query->countQuery()->execute()->fetchField();
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $data[] = [
          'nid' => $row->nid,
          'page_name' => $row->page_type_name ?? '',
          'trees_name' => $row->treesname ?? '',
          'count_tress' => $row->trees_count ?? '',
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      $total_pages = ceil($total / $limit);

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Tress Data fetched successfully',
        'data' => [
          'items' => $data,
          'pagination' => [
            'total' => (int)$total,
            'page' => $page,
            'limit' => $limit,
            'total_pages' => $total_pages
          ]
        ]
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  public function OtherDetailsPlanationDriveInnerPage(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $page_type = $content['page_type'] ?? '';

      if (empty($requested_langcode) || empty($page_type)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($page_type) ? 'Page Type is required.' : 'Language code (lan) is required.',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode, $page_type) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'plantation_drive_other_details')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        // Add a JOIN safely
        $query->join('node__field_select_pd', 'pd', 'pd.entity_id = nfd.nid');
        $query->condition('pd.field_select_pd_target_id', $page_type);

        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $page_type);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $page_type);
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $intro_text = $node->get('field_intro_text')->value ?? '';
        $location = $node->get('field_location')->value ?? '';
        $location_link = $node->get('field_location_link')->value ?? '';
        $description = $node->get('field_pd_desc')->value ?? '';
        $plantation_date = $node->get('field_plantation_date')->value ?? '';
        $tree_species = $node->get('field_trees_species')->value ?? '';
        $total_trees = $node->get('field_total_trees')->value ?? '';
        $plantation_drive_tid = $node->get('field_select_pd')->target_id ?? ''; // --> This is a taxonomy reference

        $plantation_drive = '';
        if (!empty($plantation_drive_tid)) {
          $term = \Drupal\taxonomy\Entity\Term::load($plantation_drive_tid);
          if ($term) {
            $plantation_drive = $term->getName(); // This gets the term name
          }
        }

        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'plantation_drive' => $plantation_drive,
          'intro_text' => $intro_text,
          'location' => $location,
          'location_link' => $location_link,
          'description' => $description,
          'plantation_date' => $plantation_date,
          'total_trees' => $total_trees,
          'tree_species' => $tree_species
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  public function getTaxonomyFieldsData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';
      $page_tid = $content['page_type'] ?? '';

      if (empty($langcode) || empty($page_tid)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($page_tid) ? 'Page Type is required.' : 'Language code (lan) is required.',
        ], 400);
      }

      $term = \Drupal\taxonomy\Entity\Term::load($page_tid);
      if (!$term) {
        return new JsonResponse([
          'status' => false,
          'message' => 'No term found with the given ID.',
        ], 404);
      }

      $fallback = false;

      if ($term->hasTranslation($langcode)) {
        $translated_term = $term->getTranslation($langcode);
      } else {
        $translated_term = $term->hasTranslation('en') ? $term->getTranslation('en') : $term;
        $langcode = 'en';
        $fallback = true;
      }

      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      $term_name = $translated_term->getName();
      $menu_link = $translated_term->get('field_menu_link')->value ?? '';

      // Handle multi-value text field
      $texts = [];
      foreach ($translated_term->get('field_extra_texts')->getValue() as $item) {
        if (!empty($item['value'])) {
          $texts[] = $item['value'];
        }
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback ? 'Fallback to English. Translation not available in requested language.' : 'Term fetched successfully.',
        'data' => [
          'tid' => $translated_term->id(),
          'term_name' => $term_name,
          'language_code' => $langcode,
          'language_name' => $language_name,
          'menu_link' => $menu_link,
          'extra_texts' => $texts,
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
}
