<?php

namespace Drupal\api_module\Controller;

use Exception;
use Drupal\Core\Cache\Cache;
use Drupal\taxonomy\Entity\Term;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;

class GenericFormulationController extends ControllerBase
{
  public function getGenericFormulationProducts(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';
      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $page = max(1, (int)($content['page'] ?? 1));
      $limit = max(1, min(100, (int)($content['limit'] ?? 10)));
      $offset = ($page - 1) * $limit;

      // Get language object
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      // Build base query
      $query = \Drupal::entityQuery('node')
        ->condition('type', 'generic_formulations_products')
        ->condition('status', 1)
        ->accessCheck(true);

      $nids = $query->execute();
      if (empty($nids)) {
        return new JsonResponse([
          'status' => true,
          'message' => 'No results found',
          'data' => [],
          'pagination' => [
            'current_page' => $page,
            'total_pages' => 0,
            'total_items' => 0,
            'items_per_page' => $limit,
          ],
        ], 200);
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);

      // --- STEP 1: Collect only translated nodes for requested lang ---
      $translated_nodes = [];
      foreach ($nodes as $node) {
        if ($node->hasTranslation($langcode)) {
          $translated_nodes[] = $node->getTranslation($langcode);
        }
      }

      $fallback_used = false;

      // --- STEP 2: If no translations found, fallback to English ---
      if (empty($translated_nodes) && $langcode !== 'en') {
        foreach ($nodes as $node) {
          if ($node->hasTranslation('en')) {
            $translated_nodes[] = $node->getTranslation('en');
          } else {
            $translated_nodes[] = $node; // raw if no en translation
          }
        }
        $fallback_used = true;
        $language_name = "English";
      }

      // --- Pagination ---
      $total = count($translated_nodes);
      $paged_nodes = array_slice($translated_nodes, $offset, $limit);

      // --- Taxonomy preload ---
      $taxonomy_fields = [
        'field_dossier_status' => 'dossier_status',
        'field_therapy_area_gf' => 'therapy_area',
      ];

      $term_ids = [];
      foreach ($paged_nodes as $node) {
        foreach ($taxonomy_fields as $field => $key) {
          if ($node->hasField($field) && !$node->get($field)->isEmpty()) {
            $term_ids[$field][] = $node->get($field)->target_id;
          }
        }
      }

      $terms = [];
      foreach ($term_ids as $field => $ids) {
        $loaded_terms = \Drupal::entityTypeManager()
          ->getStorage('taxonomy_term')
          ->loadMultiple(array_unique($ids));
        $terms[$field] = [];
        foreach ($loaded_terms as $tid => $term) {
          if ($term->hasTranslation($langcode)) {
            $terms[$field][$tid] = $term->getTranslation($langcode)->label();
          } elseif ($langcode !== 'en' && $term->hasTranslation('en')) {
            $terms[$field][$tid] = $term->getTranslation('en')->label(); // fallback
          } else {
            $terms[$field][$tid] = $term->label(); // raw if no translation at all
          }
        }
      }

      // --- Final data build ---
      $formulations = [];
      foreach ($paged_nodes as $node) {
        $formulation = [
          'nid' => $node->id(),
          'product_name' => $node->hasField('field_product_name_gf') ? $node->get('field_product_name_gf')->value : '',
          'strength' => $node->hasField('field_strength') ? $node->get('field_strength')->value : '',
          'inj' => $node->hasField('field_inj') ? $node->get('field_inj')->value : '',
          'oral' => $node->hasField('field_oral') ? $node->get('field_oral')->value : '',
          'language' => $language_name,
        ];

        foreach ($taxonomy_fields as $field => $key) {
          $formulation[$key] = '';
          if ($node->hasField($field) && !$node->get($field)->isEmpty()) {
            $tid = $node->get($field)->target_id;
            $formulation[$key] = $terms[$field][$tid] ?? '';
          }
        }

        $formulations[] = $formulation;
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback_used
          ? 'Fallback to English as no content found in requested language.'
          : 'Generic formulations retrieved successfully',
        'data' => $formulations,
        'pagination' => [
          'current_page' => $page,
          'total_pages' => ceil($total / $limit),
          'total_items' => $total,
          'items_per_page' => $limit,
        ],
      ], 200);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => $e->getMessage(),
      ], $e->getCode() ?: 500);
    }
  }

  // public function getGenericFormulationClinicalProducts(Request $request)
  // {
  //   try {
  //     // Decode JSON content once and validate
  //     $content = json_decode($request->getContent(), true);
  //     if (!is_array($content)) {
  //       throw new \Exception('Invalid JSON content', 400);
  //     }

  //     // Extract and validate inputs
  //     $langcode = $content['lan'] ?? '';
  //     if (empty($langcode)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => 'Language code (lan) is required',
  //       ], 400);
  //     }

  //     $page = max(1, (int)($content['page'] ?? 1));
  //     $limit = max(1, min(100, (int)($content['limit'] ?? 10)));
  //     $offset = ($page - 1) * $limit;

  //     // Get language name (cached by Drupal's LanguageManager)
  //     $language = \Drupal::languageManager()->getLanguage($langcode);
  //     $language_name = $language ? $language->getName() : $langcode;

  //     // Build base entity query
  //     $query = \Drupal::entityQuery('node')
  //       ->condition('type', 'generic_formulation_clinical_tri')
  //       ->condition('status', 1)
  //       ->accessCheck(true);

  //     // Clone query for total count
  //     $total = (clone $query)->count()->execute();

  //     // Add pagination to main query
  //     $nids = $query->range($offset, $limit)->execute();
  //     if (empty($nids)) {
  //       return new JsonResponse([
  //         'status' => true,
  //         'message' => 'No results found',
  //         'data' => [],
  //         'pagination' => [
  //           'current_page' => $page,
  //           'total_pages' => 0,
  //           'total_items' => 0,
  //           'items_per_page' => $limit,
  //         ],
  //       ], 200);
  //     }

  //     // Load nodes and taxonomy terms in bulk
  //     $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
  //     $formulations = [];
  //     $taxonomy_fields = [
  //       'field_dosage_form' => 'dosage_form',
  //       'field_eu_approval_status' => 'eu_approval_status',
  //       'field_us_approval_status' => 'us_approval_status',
  //       'field_therapy_area_gf_ct' => 'therapy_area',
  //     ];

  //     // Preload taxonomy terms to reduce queries
  //     $term_ids = [];
  //     foreach ($nodes as $node) {
  //       foreach ($taxonomy_fields as $field => $key) {
  //         if ($node->hasField($field) && !$node->get($field)->isEmpty()) {
  //           $term_ids[$field][] = $node->get($field)->target_id;
  //         }
  //       }
  //     }

  //     // Load all taxonomy terms in bulk
  //     $terms = [];
  //     foreach ($term_ids as $field => $ids) {
  //       $loaded_terms = \Drupal::entityTypeManager()
  //         ->getStorage('taxonomy_term')
  //         ->loadMultiple(array_unique($ids));
  //       $terms[$field] = array_combine(
  //         array_keys($loaded_terms),
  //         array_map(fn($term) => $term->label(), $loaded_terms)
  //       );
  //     }

  //     // Process nodes
  //     foreach ($nodes as $node) {
  //       $formulation = [
  //         'nid' => $node->id(),
  //         'product_name' => $node->hasField('field_product_name_gf_ct') ? $node->get('field_product_name_gf_ct')->value : '',
  //         'language' => $language_name,
  //       ];

  //       // Assign taxonomy term names
  //       foreach ($taxonomy_fields as $field => $key) {
  //         $formulation[$key] = '';
  //         if ($node->hasField($field) && !$node->get($field)->isEmpty()) {
  //           $tid = $node->get($field)->target_id;
  //           $formulation[$key] = $terms[$field][$tid] ?? '';
  //         }
  //       }
  //       $formulations[] = $formulation;
  //     }

  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => 'Generic formulations clinical trials retrieved successfully',
  //       'data' => $formulations,
  //       'pagination' => [
  //         'current_page' => $page,
  //         'total_pages' => ceil($total / $limit),
  //         'total_items' => (int)$total,
  //         'items_per_page' => $limit,
  //       ],
  //     ], 200);
  //   } catch (\Exception $e) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => $e->getMessage(),
  //     ], $e->getCode() ?: 500);
  //   }
  // }
  public function getGenericFormulationClinicalProducts(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';
      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $page = max(1, (int)($content['page'] ?? 1));
      $limit = max(1, min(100, (int)($content['limit'] ?? 10)));
      $offset = ($page - 1) * $limit;

      // Get language object
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      // Build base query
      $query = \Drupal::entityQuery('node')
        ->condition('type', 'generic_formulation_clinical_tri')
        ->condition('status', 1)
        ->accessCheck(true);

      $nids = $query->execute();
      if (empty($nids)) {
        return new JsonResponse([
          'status' => true,
          'message' => 'No results found',
          'data' => [],
          'pagination' => [
            'current_page' => $page,
            'total_pages' => 0,
            'total_items' => 0,
            'items_per_page' => $limit,
          ],
        ], 200);
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);

      // --- STEP 1: Collect only translated nodes for requested lang ---
      $translated_nodes = [];
      foreach ($nodes as $node) {
        if ($node->hasTranslation($langcode)) {
          $translated_nodes[] = $node->getTranslation($langcode);
        }
      }

      $fallback_used = false;

      // --- STEP 2: If no translations found, fallback to English ---
      if (empty($translated_nodes) && $langcode !== 'en') {
        foreach ($nodes as $node) {
          if ($node->hasTranslation('en')) {
            $translated_nodes[] = $node->getTranslation('en');
          } else {
            $translated_nodes[] = $node; // raw if no en translation
          }
        }
        $fallback_used = true;
        $language_name = "English";
      }

      // --- Pagination ---
      $total = count($translated_nodes);
      $paged_nodes = array_slice($translated_nodes, $offset, $limit);

      // --- Taxonomy preload ---
      $taxonomy_fields = [
        'field_dosage_form' => 'dosage_form',
        'field_eu_approval_status' => 'eu_approval_status',
        'field_us_approval_status' => 'us_approval_status',
        'field_therapy_area_gf_ct' => 'therapy_area',
      ];

      $term_ids = [];
      foreach ($paged_nodes as $node) {
        foreach ($taxonomy_fields as $field => $key) {
          if ($node->hasField($field) && !$node->get($field)->isEmpty()) {
            $term_ids[$field][] = $node->get($field)->target_id;
          }
        }
      }

      $terms = [];
      foreach ($term_ids as $field => $ids) {
        $loaded_terms = \Drupal::entityTypeManager()
          ->getStorage('taxonomy_term')
          ->loadMultiple(array_unique($ids));
        $terms[$field] = [];
        foreach ($loaded_terms as $tid => $term) {
          if ($term->hasTranslation($langcode)) {
            $terms[$field][$tid] = $term->getTranslation($langcode)->label();
          } elseif ($langcode !== 'en' && $term->hasTranslation('en')) {
            $terms[$field][$tid] = $term->getTranslation('en')->label(); // fallback
          } else {
            $terms[$field][$tid] = $term->label(); // raw if no translation at all
          }
        }
      }

      // --- Final data build ---
      $formulations = [];
      foreach ($paged_nodes as $node) {
        $formulation = [
          'nid' => $node->id(),
          'product_name' => $node->hasField('field_product_name_gf_ct') ? $node->get('field_product_name_gf_ct')->value : '',
          'language' => $language_name,
        ];

        foreach ($taxonomy_fields as $field => $key) {
          $formulation[$key] = '';
          if ($node->hasField($field) && !$node->get($field)->isEmpty()) {
            $tid = $node->get($field)->target_id;
            $formulation[$key] = $terms[$field][$tid] ?? '';
          }
        }

        $formulations[] = $formulation;
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback_used
          ? 'Fallback to  English as no content found in requested language.'
          : 'Generic formulations clinical trials retrieved successfully',
        'data' => $formulations,
        'pagination' => [
          'current_page' => $page,
          'total_pages' => ceil($total / $limit),
          'total_items' => $total,
          'items_per_page' => $limit,
        ],
      ], 200);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => $e->getMessage(),
      ], $e->getCode() ?: 500);
    }
  }

  public function getGenericFormulationSearchKeys(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';

      // Validate language code
      if (empty($langcode)) {
        return new JsonResponse(['status' => false, 'message' => 'Language code (lan) is required'], 400);
      }

      $language = \Drupal::languageManager()->getLanguage($langcode);
      if (!$language) {
        return new JsonResponse(['status' => false, 'message' => 'Invalid language code provided'], 400);
      }

      // Check cache
      $cache_id = "taxonomy_terms:$langcode";
      $cache = \Drupal::cache()->get($cache_id);
      if ($cache) {
        return new JsonResponse($cache->data, 200);
      }

      // Fetch taxonomy terms
      $therapy_tids = \Drupal::entityQuery('taxonomy_term')
        ->condition('vid', 'therapy_area_generic_formulation')
        ->accessCheck(TRUE)
        ->execute();

      $therapy_areas = [];
      if (!empty($therapy_tids)) {
        $therapy_terms = \Drupal::entityTypeManager()
          ->getStorage('taxonomy_term')
          ->loadMultiple($therapy_tids);

        foreach ($therapy_terms as $term) {
          $translated_term = $term->hasTranslation($langcode) ? $term->getTranslation($langcode) : $term;
          $therapy_areas[] = [
            'tid' => $term->id(),
            'name' => $translated_term->getName() ?: 'Unnamed Term',
          ];
        }
      }

      // Prepare response
      $response = [
        'status' => true,
        'data' => [
          'therapy_area_generic_formulation' => $therapy_areas,
        ],
      ];

      // Store in cache with taxonomy term cache tags
      \Drupal::cache()->set($cache_id, $response, Cache::PERMANENT, ['taxonomy_term_list']);

      return new JsonResponse($response, 200);
    } catch (\Exception $e) {
      \Drupal::logger('taxonomy_api')->error('Error fetching taxonomy terms: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['status' => false, 'message' => 'An error occurred'], 500);
    }
  }

  public function getGenericFormulationCapabilities(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Query builder function
      $fetchContent = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_gfpc_description', 'nfdes', 'nfdes.entity_id = nfd.nid AND nfdes.delta = 0 AND nfdes.langcode = :langcode');
        $query->addField('nfdes', 'field_gfpc_description_value', 'GFPC_description');

        $query->leftJoin('node__field_gfpc_description2', 'ndes2', 'ndes2.entity_id = nfd.nid AND ndes2.delta = 0 AND ndes2.langcode = :langcode');
        $query->addField('ndes2', 'field_gfpc_description2_value', 'gfpc_description2');

        $query->leftJoin('node__field_product_capability_super_h', 'nfsico', 'nfsico.entity_id = nfd.nid AND nfsico.delta = 0 AND nfsico.langcode = :langcode');
        $query->addField('nfsico', 'field_product_capability_super_h_value', 'capability_super_heading');

        $query->leftJoin('node__field_tabular_data', 'nftabu', 'nftabu.entity_id = nfd.nid AND nftabu.delta = 0 AND nftabu.langcode = :langcode');
        $query->addField('nftabu', 'field_tabular_data_value', 'tabular_dat');

        $query->condition('nfd.type', 'generic_formulation_capabilities');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);

        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Step 1: Fetch requested language
      $results = $fetchContent($requested_langcode);

      // Step 2: Fallback to English
      $fallback = false;
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchContent('en');
        $fallback = true;
      }

      // Language names for additional metadata (optional)
      $lang_names = \Drupal::languageManager()->getStandardLanguageList();

      $data = [];

      foreach ($results as $row) {
        $nid = $row->nid;

        $data[] = [
          'nid' => $nid,
          'title' => $row->title,
          'capability_super_heading' => $row->capability_super_heading ?? '',
          'GFPC_description' => $row->GFPC_description ?? '',
          'gfpc_description2' => $row->gfpc_description2 ?? '',
          'tabular_data' => $row->tabular_dat ?? '',
          'langcode' => $row->langcode,
          'language_name' => isset($lang_names[$row->langcode]) ? $lang_names[$row->langcode][0] : ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'Content data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getallGenericTopSection(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'generic_formulation_main_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $heading = $node->get('field_gfms_heading')->value ?? '';
        $desc = $node->get('field_gfms_description')->value ?? '';
        $file_id = $node->get('field_gfms_brochure')->target_id ?? '';

        $file = "";
        if (!empty($file_id)) {
          $temp = \Drupal\file\Entity\File::load($file_id);
          if ($temp) {
            $file = \Drupal::service('file_url_generator')->generateAbsoluteString($temp->getFileUri());
          }
        }

        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'heading' => $heading,
          'description' => $desc,
          'brochure_url' => $file
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
}
