<?php

namespace Drupal\api_module\Controller;

use Exception;
use Drupal\file\Entity\File;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\media\Entity\Media;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;

class EventController extends ControllerBase
{
  public function getAllEventContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $langcode = $content['lan'] ?? '';
      $page = $content['page'] ?? '';

      if (empty($langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      // Get language name
      $language = \Drupal::languageManager()->getLanguage($langcode);
      $language_name = $language ? $language->getName() : $langcode;

      $current_time = new \DateTime();
      $start_of_month = strtotime($current_time->format('Y-m-01'));
      $end_of_month = strtotime($current_time->format('Y-m-t'));
      $now = time();

      $nids = \Drupal::entityQuery('node')
        ->condition('type', 'event_section')
        ->condition('status', 1)
        ->accessCheck(true)
        ->execute();

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);

      $events_this_month = [];
      $future_events = [];
      $past_events = [];

      foreach ($nodes as $node) {
        // Translate node if available
        if ($node->hasTranslation($langcode)) {
          $node = $node->getTranslation($langcode);
        }

        $start_date = $node->get('field_start_date')->value ?? '';
        $end_date = $node->get('field_end_date')->value ?? '';

        if (!$start_date || !$end_date) {
          continue;
        }

        $start_timestamp = strtotime($start_date);
        $end_timestamp = strtotime($end_date);

        // Format date like "9-11 April 2025"
        $start_day = date('j', $start_timestamp);
        $end_day = date('j', $end_timestamp);
        $month_year = date('M Y', $end_timestamp);
        $formatted_date = "$start_day-$end_day $month_year";

        $image_url = '';
        if (!$node->get('field_event_image')->isEmpty()) {
          $file = $node->get('field_event_image')->entity;
          if ($file) {
            $image_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $event = [
          'nid' => $node->id(),
          'langcode' => $langcode,
          'language_name' => $language_name,
          'title' => $node->get('field_event_title')->value ?? '',
          'event_link' => $node->get('field_event_link')->value ?? '',
          'image' => $image_url,
          'location' => $node->get('field_event_location')->value ?? '',
          'date_range' => $formatted_date,
        ];

        if ($end_timestamp < $now) {
          $past_events[] = $event;
        } elseif ($start_timestamp >= $start_of_month && $start_timestamp <= $end_of_month) {
          $events_this_month[] = $event;
        } else {
          $future_events[] = $event;
        }
      }

      // Prepare response based on `page`
      if ($page === 'home') {
        // Only upcoming events (this month + future)
        $all_events = array_merge($events_this_month, $future_events);
      } else {
        // All events (this month + future + past)
        $all_events = array_merge($events_this_month, $future_events, $past_events);
      }

      return new JsonResponse([
        'data' => $all_events,
        'status' => true,
        'message' => 'Data fetched successfully',
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'data' => [],
        'status' => false,
        'message' => 'Error: ' . $e->getMessage(),
      ], 500);
    }
  }
  public function getEventDetailsContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $event_link = $content['event_link'] ?? '';

      if (empty($requested_langcode) || empty($event_link)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($requested_langcode) ? 'Language code (lan) is required' : 'Event link is required',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode, $event_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'custom_event_details')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
        $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

        $query->condition('tfml.field_menu_link_value', $event_link);
        return $query->execute()->fetchCol();
      };

      $refineURL = function ($text) {
        $request = \Drupal::request();
        $domain = $request->getSchemeAndHttpHost();
        $text = preg_replace_callback(
          '/src=["\']\/([^"\']*)["\']/',
          function ($matches) use ($domain) {
            return 'src="' . $domain . '/' . ltrim($matches[1], '/') . '"';
          },
          $text
        );

        return $text;
      };

      $nids = $fetchNodeIds($requested_langcode, $event_link);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $event_link);
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();

        $add_our_teams = $node->get('field_add_our_teams_section')->value ?? 0;
        $add_our_achievement = $node->get('field_add_our_achievements_award')->value ?? 0;
        $add_gallery = $node->get('field_add_gallery_section')->value ?? 0;


        $event_logo_tid = $node->get('field_evnt_logo')->target_id ?? '';
        $event_banner_tid = $node->get('field_evnt_banner_image')->target_id ?? '';
        $event_title = $node->get('field_evnt_title')->value ?? '';
        $event_subheading = $node->get('field_evnt_subheading')->value ?? '';
        $event_location = $node->get('field_evnt_location')->value ?? '';
        $event_meet_us_at = $node->get('field_event_meet_us_at')->value ?? '';
        $start_date = $node->get('field_event_start_date')->value ?? '';
        $end_date = $node->get('field_event_end_date')->value ?? '';

        $section_1_label = $node->get('field_section_1_label')->value ?? '';
        $section_1_desc = $node->get('field_section_1_description')->value ?? '';
        $section_1_css = $node->get('field_section_1_css')->value ?? '';

        $section_2_label = $node->get('field_section_2_label')->value ?? '';
        $section_2_desc = $node->get('field_section_2_description')->value ?? '';
        $section_2_css = $node->get('field_section_2_css')->value ?? '';

        $section_3_label = $node->get('field_section_3_label')->value ?? '';
        $section_3_desc = $node->get('field_section_3_description')->value ?? '';
        $section_3_css = $node->get('field_section_3_css')->value ?? '';

        $section_4_label = $node->get('field_section_4_label')->value ?? '';
        $section_4_desc = $node->get('field_section_4_description')->value ?? '';
        $section_4_css = $node->get('field_section_4_css')->value ?? '';

        $section_5_label = $node->get('field_section_5_label')->value ?? '';
        $section_5_desc = $node->get('field_section_5_description')->value ?? '';
        $section_5_css = $node->get('field_section_5_css')->value ?? '';

        $section_6_label = $node->get('field_section_6_label')->value ?? '';
        $section_6_desc = $node->get('field_section_6_description')->value ?? '';
        $section_6_css = $node->get('field_section_6_css')->value ?? '';

        $section_7_label = $node->get('field_section_7_label')->value ?? '';
        $section_7_desc = $node->get('field_section_7_description')->value ?? '';
        $section_7_css = $node->get('field_section_7_css')->value ?? '';

        $event_logo = '';
        $event_banner = '';
        if (!empty($event_logo_tid)) {
          $file = \Drupal\file\Entity\File::load($event_logo_tid);
          if ($file) {
            $event_logo = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        if (!empty($event_banner_tid)) {
          $file = \Drupal\file\Entity\File::load($event_banner_tid);
          if ($file) {
            $event_banner = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $label_data = [];
        $sections_data = [];
        $label_data[0]['id'] = "#section-1";
        $label_data[0]['label'] = $section_1_label;
        $sections_data[0]['id'] =  "#section-1";
        $sections_data[0]['label'] = $section_1_label;
        $sections_data[0]['description'] = $section_1_desc;
        $sections_data[0]['css'] = $section_1_css;

        $label_data[1]['id'] = "#section-2";
        $label_data[1]['label'] = $section_2_label;
        $sections_data[1]['id'] = "#section-2";
        $sections_data[1]['label'] = $section_2_label;
        $sections_data[1]['description'] = $section_2_desc;
        $sections_data[1]['css'] = $section_2_css;

        $label_data[2]['id'] = "#section-3";
        $label_data[2]['label'] = $section_3_label;
        $sections_data[2]['id'] = "#section-3";
        $sections_data[2]['label'] = $section_3_label;
        $sections_data[2]['description'] = $section_3_desc;
        $sections_data[2]['css'] = $section_3_css;

        $label_data[3]['id'] = "#section-4";
        $label_data[3]['label'] = $section_4_label;
        $sections_data[3]['id'] = "#section-4";
        $sections_data[3]['label'] = $section_4_label;
        $sections_data[3]['description'] = $section_4_desc;
        $sections_data[3]['css'] = $section_4_css;

        $label_data[4]['id'] = "#section-5";
        $label_data[4]['label'] = $section_5_label;
        $sections_data[4]['id'] = "#section-5";
        $sections_data[4]['label'] = $section_5_label;
        $sections_data[4]['description'] = $section_5_desc;
        $sections_data[4]['css'] = $section_5_css;

        $label_data[5]['id'] = "#section-6";
        $label_data[5]['label'] = $section_6_label;
        $sections_data[5]['id'] = "#section-6";
        $sections_data[5]['label'] = $section_6_label;
        $sections_data[5]['description'] = $section_6_desc;
        $sections_data[5]['css'] = $section_6_css;

        $label_data[6]['id'] = "#section-7";
        $label_data[6]['label'] = $section_7_label;
        $sections_data[6]['id'] = "#section-7";
        $sections_data[6]['label'] = $section_7_label;
        $sections_data[6]['description'] = $section_7_desc;
        $sections_data[6]['css'] = $section_7_css;

        foreach ($sections_data as &$data) {
          $data['description'] = $refineURL($data['description']);
          $data['content'] = [];
        }
        unset($data);

        if ($add_our_teams === '1') {
          $OurTeamsData = $this->getEventTeams_helper($event_link, $requested_langcode);
          $index = 6  + (int)$OurTeamsData['data']['order'];
          // $add_our_teams_id = "#section-" . ($index + 1);
          $add_our_teams_id = "#ourteamssection";
          $label_data[$index]['id'] = $add_our_teams_id;
          $label_data[$index]['label'] = $OurTeamsData['data']['label'];

          $sections_data[$index]['id'] = $add_our_teams_id;
          $sections_data[$index]['label'] = $OurTeamsData['data']['label'];
          $sections_data[$index]['description'] = "";
          $sections_data[$index]['css'] = "";
          $sections_data[$index]['content'] = $OurTeamsData['data'];
        }

        if ($add_gallery === '1') {
          $OurGalleryData = $this->getEventGallery_helper($event_link, $requested_langcode);
          $index = 6  + (int)$OurGalleryData['data']['order'];
          // $add_gallery_id = "#section-" . ($index + 1);
          $add_gallery_id = "#gallerysection";
          $label_data[$index]['id'] = $add_gallery_id;
          $label_data[$index]['label'] = $OurGalleryData['data']['label'];

          $sections_data[$index]['id'] = $add_gallery_id;
          $sections_data[$index]['label'] = $OurGalleryData['data']['label'];
          $sections_data[$index]['description'] = "";
          $sections_data[$index]['css'] = "";
          $sections_data[$index]['content'] = $OurGalleryData['data'];
        }

        if ($add_our_achievement === '1') {
          $EventAchievementData = $this->getEventAcheivement_helper($event_link, $requested_langcode);
          $index = 6  + (int)$EventAchievementData['order'];
          // $add_achievement_id = "#section-" . ($index + 1);
          $add_achievement_id = "#ourachievementsection";
          $label_data[$index]['id'] = $add_achievement_id;
          $label_data[$index]['label'] = $EventAchievementData['label'];

          $sections_data[$index]['id'] = $add_achievement_id;
          $sections_data[$index]['label'] = $EventAchievementData['label'];
          $sections_data[$index]['description'] = "";
          $sections_data[$index]['css'] = "";
          $sections_data[$index]['content'] = $EventAchievementData['data'];
        }
        $data = [
          'nid' => $nid,
          'event_banner' => $event_banner ?? '',
          'event_logo' => $event_logo ?? '',
          'event_title' => $event_title,
          'event_subheading' => $event_subheading,
          'event_meet_us_at' => $event_meet_us_at,
          'event_location' => $event_location,
          'start_date' => $start_date,
          'end_date' => $end_date,
          'tab_labels' => !empty($label_data) ? $label_data : [],
          'sections' => !empty($sections_data) ? $sections_data : [],
          'langcode' => $requested_langcode,
          'language_name' => $language_name
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Data fetched successfully',
        'data' => $data

      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  // public function getEventDetailsContent(Request $request)
  // {
  //   try {
  //     $content = json_decode($request->getContent(), true);
  //     $requested_langcode = $content['lan'] ?? '';
  //     $event_link = $content['event_link'] ?? '';

  //     if (empty($requested_langcode) || empty($event_link)) {
  //       return new JsonResponse([
  //         'status' => false,
  //         'message' => empty($requested_langcode) ? 'Language code (lan) is required' : 'Event link is required',
  //       ], 400);
  //     }

  //     $database = \Drupal::database();

  //     $fetchNodeIds = function ($langcode, $event_link) use ($database) {
  //       $query = $database->select('node_field_data', 'nfd')
  //         ->fields('nfd', ['nid'])
  //         ->condition('nfd.type', 'custom_event_details')
  //         ->condition('nfd.status', 1)
  //         ->condition('nfd.langcode', $langcode);

  //       $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
  //       $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

  //       $query->condition('tfml.field_menu_link_value', $event_link);
  //       return $query->execute()->fetchCol();
  //     };

  //     $refineURL = function ($text) {
  //       $request = \Drupal::request();
  //       $domain = $request->getSchemeAndHttpHost();
  //       $text = preg_replace_callback(
  //         '/src=["\']\/([^"\']*)["\']/',
  //         function ($matches) use ($domain) {
  //           return 'src="' . $domain . '/' . ltrim($matches[1], '/') . '"';
  //         },
  //         $text
  //       );

  //       return $text;
  //     };

  //     $nids = $fetchNodeIds($requested_langcode, $event_link);
  //     $fallback = false;

  //     // Fallback to English if no nodes found
  //     if (empty($nids) && $requested_langcode !== 'en') {
  //       $requested_langcode = 'en';
  //       $nids = $fetchNodeIds('en', $event_link);
  //       $fallback = true;
  //     }

  //     $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
  //     $language = \Drupal::languageManager()->getLanguage($requested_langcode);
  //     $language_name = $language ? $language->getName() : $requested_langcode;

  //     foreach ($nodes as $node) {
  //       if ($node->hasTranslation($requested_langcode)) {
  //         $node = $node->getTranslation($requested_langcode);
  //       }

  //       $nid = $node->id();
  //       $event_logo_tid = $node->get('field_evnt_logo')->target_id ?? '';
  //       $event_banner_tid = $node->get('field_evnt_banner_image')->target_id ?? '';
  //       $event_title = $node->get('field_evnt_title')->value ?? '';
  //       $event_subheading = $node->get('field_evnt_subheading')->value ?? '';
  //       $event_location = $node->get('field_evnt_location')->value ?? '';
  //       $event_meet_us_at = $node->get('field_event_meet_us_at')->value ?? '';
  //       $start_date = $node->get('field_event_start_date')->value ?? '';
  //       $end_date = $node->get('field_event_end_date')->value ?? '';

  //       $section_1_label = $node->get('field_section_1_label')->value ?? '';
  //       $section_1_desc = $node->get('field_section_1_description')->value ?? '';
  //       $section_1_css = $node->get('field_section_1_css')->value ?? '';

  //       $section_2_label = $node->get('field_section_2_label')->value ?? '';
  //       $section_2_desc = $node->get('field_section_2_description')->value ?? '';
  //       $section_2_css = $node->get('field_section_2_css')->value ?? '';

  //       $section_3_label = $node->get('field_section_3_label')->value ?? '';
  //       $section_3_desc = $node->get('field_section_3_description')->value ?? '';
  //       $section_3_css = $node->get('field_section_3_css')->value ?? '';

  //       $section_4_label = $node->get('field_section_4_label')->value ?? '';
  //       $section_4_desc = $node->get('field_section_4_description')->value ?? '';
  //       $section_4_css = $node->get('field_section_4_css')->value ?? '';

  //       $section_5_label = $node->get('field_section_5_label')->value ?? '';
  //       $section_5_desc = $node->get('field_section_5_description')->value ?? '';
  //       $section_5_css = $node->get('field_section_5_css')->value ?? '';

  //       $section_6_label = $node->get('field_section_6_label')->value ?? '';
  //       $section_6_desc = $node->get('field_section_6_description')->value ?? '';
  //       $section_6_css = $node->get('field_section_6_css')->value ?? '';

  //       $section_7_label = $node->get('field_section_7_label')->value ?? '';
  //       $section_7_desc = $node->get('field_section_7_description')->value ?? '';
  //       $section_7_css = $node->get('field_section_7_css')->value ?? '';

  //       $event_logo = '';
  //       $event_banner = '';
  //       if (!empty($event_logo_tid)) {
  //         $file = \Drupal\file\Entity\File::load($event_logo_tid);
  //         if ($file) {
  //           $event_logo = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
  //         }
  //       }
  //       if (!empty($event_banner_tid)) {
  //         $file = \Drupal\file\Entity\File::load($event_banner_tid);
  //         if ($file) {
  //           $event_banner = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
  //         }
  //       }

  //       $label_data = [];
  //       $sections_data = [];
  //       $label_data[0]['id'] = "#section-1";
  //       $label_data[0]['label'] = $section_1_label;
  //       $sections_data[0]['id'] =  "#section-1";
  //       $sections_data[0]['label'] = $section_1_label;
  //       $sections_data[0]['description'] = $section_1_desc;
  //       $sections_data[0]['css'] = $section_1_css;

  //       $label_data[1]['id'] = "#section-2";
  //       $label_data[1]['label'] = $section_2_label;
  //       $sections_data[1]['id'] = "#section-2";
  //       $sections_data[1]['label'] = $section_2_label;
  //       $sections_data[1]['description'] = $section_2_desc;
  //       $sections_data[1]['css'] = $section_2_css;

  //       $label_data[2]['id'] = "#section-3";
  //       $label_data[2]['label'] = $section_3_label;
  //       $sections_data[2]['id'] = "#section-3";
  //       $sections_data[2]['label'] = $section_3_label;
  //       $sections_data[2]['description'] = $section_3_desc;
  //       $sections_data[2]['css'] = $section_3_css;

  //       $label_data[3]['id'] = "#section-4";
  //       $label_data[3]['label'] = $section_4_label;
  //       $sections_data[3]['id'] = "#section-4";
  //       $sections_data[3]['label'] = $section_4_label;
  //       $sections_data[3]['description'] = $section_4_desc;
  //       $sections_data[3]['css'] = $section_4_css;

  //       $label_data[4]['id'] = "#section-5";
  //       $label_data[4]['label'] = $section_5_label;
  //       $sections_data[4]['id'] = "#section-5";
  //       $sections_data[4]['label'] = $section_5_label;
  //       $sections_data[4]['description'] = $section_5_desc;
  //       $sections_data[4]['css'] = $section_5_css;

  //       $label_data[5]['id'] = "#section-6";
  //       $label_data[5]['label'] = $section_6_label;
  //       $sections_data[5]['id'] = "#section-6";
  //       $sections_data[5]['label'] = $section_6_label;
  //       $sections_data[5]['description'] = $section_6_desc;
  //       $sections_data[5]['css'] = $section_6_css;

  //       $label_data[6]['id'] = "#section-7";
  //       $label_data[6]['label'] = $section_7_label;
  //       $sections_data[6]['id'] = "#section-7";
  //       $sections_data[6]['label'] = $section_7_label;
  //       $sections_data[6]['description'] = $section_7_desc;
  //       $sections_data[6]['css'] = $section_7_css;

  //       foreach ($sections_data as &$data) {
  //         $data['description'] = $refineURL($data['description']);
  //       }
  //       unset($data); // best practice after using reference
  //       $data = [
  //         'nid' => $nid,
  //         'event_banner' => $event_banner ?? '',
  //         'event_logo' => $event_logo ?? '',
  //         'event_title' => $event_title,
  //         'event_subheading' => $event_subheading,
  //         'event_meet_us_at' => $event_meet_us_at,
  //         'event_location' => $event_location,
  //         'start_date' => $start_date,
  //         'end_date' => $end_date,
  //         'tab_labels' => !empty($label_data) ? $label_data : [],
  //         'sections' => !empty($sections_data) ? $sections_data : [],
  //         'langcode' => $requested_langcode,
  //         'language_name' => $language_name
  //       ];
  //     }

  //     return new JsonResponse([
  //       'status' => true,
  //       'message' => $fallback
  //         ? 'Fallback to English. No data found in requested language.'
  //         : 'Data fetched successfully',
  //       'data' => $data,
  //     ]);
  //   } catch (\Exception $e) {
  //     return new JsonResponse([
  //       'status' => false,
  //       'message' => 'An error occurred: ' . $e->getMessage(),
  //     ], 500);
  //   }
  // }

  public function getEventTeamsContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $event_link = $content['event_link'] ?? '';

      if (empty($requested_langcode) || empty($event_link)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($requested_langcode) ? 'Language code (lan) is required' : 'Event link is required',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode, $event_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'event_our_teams')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
        $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

        $query->condition('tfml.field_menu_link_value', $event_link);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $event_link);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $event_link);
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);


      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        $team_data = [];
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $event_name = '';

        if (!$node->get('field_select_event')->isEmpty()) {
          $term = $node->get('field_select_event')->entity;

          if ($term) {
            // Check if the taxonomy term has a translation in the requested lang
            if ($term->hasTranslation($requested_langcode)) {
              $term = $term->getTranslation($requested_langcode);
            } else {
              $term = $term->getTranslation('en');
            }
            $event_name = $term->label();
          }
        }

        if ($node->hasField('field_team_details') && !$node->get('field_team_details')->isEmpty()) {
          foreach ($node->get('field_team_details') as $media_ref) {
            /** @var \Drupal\media\Entity\Media $media */
            $media = $media_ref->entity;

            if ($media instanceof Media) {
              if ($media->hasTranslation($requested_langcode)) {
                $media = $media->getTranslation($requested_langcode);
              } elseif ($requested_langcode !== 'en' && $media->hasTranslation('en')) {
                $media = $media->getTranslation('en');
              }
              $heading = $media->get('field_media_heading')->value;
              $description = $media->get('field_media_description')->value;
              $image = '';
              if (!$media->get('field_media_image_1')->isEmpty()) {
                $file = $media->get('field_media_image_1')->entity;
                $image = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
              }

              $team_data[] = [
                'name' => $heading ?? '',
                'designation' => $description ?? '',
                'image' => $image ?? ''
              ];
            }
          }
        }
        $data = [
          'nid' => $nid ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'event_name' => $event_name ?? '',
          'team_data' => !empty($team_data) ? $team_data : []
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }
  public function getEventGalleryContent(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';
      $event_link = $content['event_link'] ?? '';

      if (empty($requested_langcode) || empty($event_link)) {
        return new JsonResponse([
          'status' => false,
          'message' => empty($requested_langcode) ? 'Language code (lan) is required' : 'Event link is required',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode, $event_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'event_gallery_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
        $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

        $query->condition('tfml.field_menu_link_value', $event_link);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $event_link);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $event_link);
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);


      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {

        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }
        $gallery_data = [];
        $nid = $node->id();
        $gallery_items = $node->get('field_gallery_image')->getValue();
        $target_ids = array_column($gallery_items, 'target_id');
        foreach ($target_ids as $id) {
          if (!empty($id)) {
            $file = \Drupal\file\Entity\File::load($id);
            if ($file) {
              $gallery_data[] = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
            }
          }
        }


        $data = [
          'nid' => $nid ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'images' => !empty($gallery_data) ? $gallery_data : []
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getEventTeams_helper($event_link, $requested_langcode)
  {
    try {
      if (empty($requested_langcode) || empty($event_link)) {
        return false;
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode, $event_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'event_our_teams')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
        $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

        $query->condition('tfml.field_menu_link_value', $event_link);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $event_link);
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $event_link);
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);


      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        $team_data = [];
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $order = $node->get('field_select_order')->value ?? 0;
        $section_label = $node->get('field_section_label')->value ?? "";
        $event_name = '';

        if (!$node->get('field_select_event')->isEmpty()) {
          $term = $node->get('field_select_event')->entity;

          if ($term) {
            if ($term->hasTranslation($requested_langcode)) {
              $term = $term->getTranslation($requested_langcode);
            } else {
              $term = $term->getTranslation('en');
            }
            $event_name = $term->label();
          }
        }

        if ($node->hasField('field_team_details') && !$node->get('field_team_details')->isEmpty()) {
          foreach ($node->get('field_team_details') as $media_ref) {
            /** @var \Drupal\media\Entity\Media $media */
            $media = $media_ref->entity;

            if ($media instanceof Media) {
              if ($media->hasTranslation($requested_langcode)) {
                $media = $media->getTranslation($requested_langcode);
              } elseif ($requested_langcode !== 'en' && $media->hasTranslation('en')) {
                $media = $media->getTranslation('en');
              }
              $heading = $media->get('field_media_heading')->value;
              $description = $media->get('field_media_description')->value;
              $image = '';
              if (!$media->get('field_media_image_1')->isEmpty()) {
                $file = $media->get('field_media_image_1')->entity;
                $image = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
              }

              $team_data[] = [
                'name' => $heading ?? '',
                'designation' => $description ?? '',
                'image' => $image ?? ''
              ];
            }
          }
        }
        $data = [
          'nid' => $nid ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'event_name' => $event_name ?? '',
          'order' => $order,
          'label' => $section_label,
          'team_data' => !empty($team_data) ? $team_data : []
        ];
      }

      return [
        'data' => $data
      ];
    } catch (\Exception $e) {
      return   $e->getMessage();
    }
  }
  public function getEventGallery_helper($event_link, $requested_langcode)
  {
    try {
      if (empty($requested_langcode) || empty($event_link)) {
        return false;
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode, $event_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'event_gallery_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
        $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

        $query->condition('tfml.field_menu_link_value', $event_link);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $event_link);

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $event_link);
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);


      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {

        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }
        $gallery_data = [];
        $nid = $node->id();
        $order = $node->get('field_select_order')->value ?? 0;
        $section_label = $node->get('field_section_label')->value ?? "";
        $gallery_items = $node->get('field_gallery_image')->getValue();
        $target_ids = array_column($gallery_items, 'target_id');
        foreach ($target_ids as $id) {
          if (!empty($id)) {
            $file = \Drupal\file\Entity\File::load($id);
            if ($file) {
              $gallery_data[] = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
            }
          }
        }

        $data = [
          'nid' => $nid ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'order' => $order,
          'label' => $section_label,
          'images' => !empty($gallery_data) ? $gallery_data : []
        ];
      }
      return [
        'data' => $data
      ];
    } catch (\Exception $e) {
      return   $e->getMessage();
    }
  }

  public function getEventAcheivement_helper($event_link, $requested_langcode)
  {
    try {
      if (empty($event_link) || empty($requested_langcode)) {
        return false;
      }
      if (empty($event_link)) {
        return false;
      }
      $database = \Drupal::database();
      $fetchNodeIds = function ($langcode, $event_link) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'event_acheivements_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        $query->join('node__field_select_event', 'nse', 'nfd.nid = nse.entity_id');
        $query->join('taxonomy_term__field_menu_link', 'tfml', 'nse.field_select_event_target_id = tfml.entity_id');

        $query->condition('tfml.field_menu_link_value', $event_link);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode, $event_link);
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en', $event_link);
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }
        $nid = $node->id();
        $order = $node->get('field_select_order')->value ?? 0;
        $section_label = $node->get('field_section_label')->value ?? "";
        $menu_links = [];

        if ($node->hasField('field_related_news') && !$node->get('field_related_news')->isEmpty()) {
          foreach ($node->get('field_related_news')->referencedEntities() as $term) {
            if ($term->hasField('field_menu_link') && !$term->get('field_menu_link')->isEmpty()) {
              $menu_links[] = $term->get('field_menu_link')->value;
            }
          }
        }
      }
      $fetchcontent = function ($menu_link, $requested_langcode) {
        $results = [];
        $vid = "news";
        $terms = \Drupal::entityTypeManager()
          ->getStorage('taxonomy_term')
          ->loadByProperties([
            'vid' => $vid,
            'field_menu_link' => $menu_link,
          ]);
        if (empty($terms)) {
          return ['results' => $results];
        }
        $term = reset($terms);

        // prefer term translation if available
        if ($term->hasTranslation($requested_langcode)) {
          $term = $term->getTranslation($requested_langcode);
        } elseif ($term->hasTranslation('en')) {
          $term = $term->getTranslation('en');
        }

        $paragraph_storage = \Drupal::entityTypeManager()->getStorage('paragraph');

        // 1) Try to fetch paragraphs created in requested language (works when editors created language-specific paragraphs)
        $query = $paragraph_storage->getQuery()
          ->condition('type', 'resource_details')
          ->condition('field_resource_name.target_id', $term->id())
          ->condition('langcode', $requested_langcode)
          ->accessCheck(FALSE);
        $pids = $query->execute();

        // 2) Fallback to English paragraphs
        if (empty($pids) && $requested_langcode !== 'en') {
          $query = $paragraph_storage->getQuery()
            ->condition('type', 'resource_details')
            ->condition('field_resource_name.target_id', $term->id())
            ->condition('langcode', 'en')
            ->accessCheck(FALSE);
          $pids = $query->execute();
        }

        // 3) Last resort: any paragraph referencing the term
        if (empty($pids)) {
          $query = $paragraph_storage->getQuery()
            ->condition('type', 'resource_details')
            ->condition('field_resource_name.target_id', $term->id())
            ->accessCheck(FALSE);
          $pids = $query->execute();
        }

        if (empty($pids)) {
          return ['results' => $results];
        }

        $paragraphs = $paragraph_storage->loadMultiple($pids);
        foreach ($paragraphs as $para) {
          // prefer an actual translation on the same paragraph entity
          if ($para->hasTranslation($requested_langcode)) {
            $p = $para->getTranslation($requested_langcode);
          } elseif ($requested_langcode !== 'en' && $para->hasTranslation('en')) {
            $p = $para->getTranslation('en');
          } else {
            // force raw entity (even if only in Spanish)
            $p = $para;
          }
          $file = $p->get('field_resource_feat_image')->entity;
          $feat_url = '';

          if ($file) {
            $feat_url = \Drupal::service('file_url_generator')
              ->generateAbsoluteString($file->getFileUri());
          }
      
          $results[] = [
            'main_heading' => $p->get('field_resource_main_heading')->value ?? '',
            'date' => $p->get('field_resource_date')->value ?? '',
            'description'  => $p->get('field_res_description')->value ?? '',
            'feat_image'   => $feat_url,
            'menu_link'    => $term->get('field_menu_link')->value ?? '',
          ];
        }
        return ['results' => $results];
      };
      // $fetchcontent = function ($menu_link, $requested_langcode) {
      //   $results = [];
      //   $vid = "news";
      //   $terms = \Drupal::entityTypeManager()
      //     ->getStorage('taxonomy_term')
      //     ->loadByProperties([
      //       'vid' => $vid,
      //       'field_menu_link' => $menu_link,
      //     ]);

      //   if (empty($terms)) {
      //     return ['results' => $results];
      //   }
      //   $term = reset($terms);
      //   $term = \Drupal::service('entity.repository')->getTranslationFromContext($term, $requested_langcode);
      //   if ($term->hasTranslation($requested_langcode)) {
      //     $term = $term->getTranslation($requested_langcode);
      //   }
      //   $paragraph_storage = \Drupal::entityTypeManager()->getStorage('paragraph');
      //   $query = $paragraph_storage->getQuery()
      //     ->condition('type', 'resource_details')
      //     ->condition('field_resource_name.target_id', $term->id())
      //     ->accessCheck(FALSE);
      //   $pids = $query->execute();

      //   if (empty($pids)) {
      //     return ['results' => $results];
      //   }

      //   $paragraphs = $paragraph_storage->loadMultiple($pids);
      //   foreach ($paragraphs as $para) {
      //     $para = \Drupal::service('entity.repository')->getTranslationFromContext($para, $requested_langcode);
      //     if ($para->hasTranslation($requested_langcode)) {
      //       $para = $para->getTranslation($requested_langcode);
      //     }
      //     $results[] = [
      //       'main_heading' => $para->get('field_resource_main_heading')->value ?? '',
      //       'description' => $para->get('field_res_description')->value ?? '',
      //       'feat_image' => $para->get('field_resource_feat_image')->entity?->createFileUrl() ?? '',
      //       'menu_link' => $term->get('field_menu_link')->value ?? '',
      //     ];
      //   }
      //   return ['results' => $results];
      // };
      $related = [
        'related_news' => []
      ];
      foreach ($menu_links as $menu_link) {
        $content_result = $fetchcontent($menu_link, $requested_langcode);

        $resources = $content_result['results'];

        if (!empty($resources)) {
          foreach ($resources as $res) {
            $related['related_news'][] = [
              'date' => $res['date'],
              'title' => $res['main_heading'],
              'url' => $res['menu_link'],
              'description' => $res['description'],
              'image' => $res['feat_image']
            ];
          }
        }
      }
      return [
        'order' =>  $order,
        'label' => $section_label,
        'data' => $related['related_news']
      ];
    } catch (\Exception $e) {
      return   $e->getMessage();
    }
  }
}
