<?php

namespace Drupal\api_module\Controller;

use Exception;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;


/**
 * Returns responses for API Module routes.
 */
class ContactController extends ControllerBase
{

  /**
   * Returns a simple page.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   */
  public function getContactData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();
      $fallback = false;

      // Step 1: Try fetching node in requested language
      $fetchContact = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode'])
          ->condition('nfd.type', 'contact_us')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode)
          ->range(0, 1)
          ->distinct();

        $query->leftJoin('node__field_phone_no', 'nfauc', 'nfauc.entity_id = nfd.nid AND nfauc.langcode = nfd.langcode AND nfauc.delta = 0');
        $query->addField('nfauc', 'field_phone_no_value', 'phone_no');

        $query->leftJoin('node__field_email', 'nfaemail', 'nfaemail.entity_id = nfd.nid AND nfaemail.langcode = nfd.langcode AND nfaemail.delta = 0');
        $query->addField('nfaemail', 'field_email_value', 'content_email');

        $query->leftJoin('node__field_address', 'nfaaddress', 'nfaaddress.entity_id = nfd.nid AND nfaaddress.langcode = nfd.langcode AND nfaaddress.delta = 0');
        $query->addField('nfaaddress', 'field_address_value', 'address');

        return $query->execute()->fetchObject();
      };

      $row = $fetchContact($requested_langcode);

      // Step 2: Fallback to English if not found
      if (!$row && $requested_langcode !== 'en') {
        $row = $fetchContact('en');
        $requested_langcode = 'en';
        $fallback = true;
      }

      if (!$row) {
        return new JsonResponse([
          'status' => true,
          'data' => [],
          'message' => 'No contact information found',
        ], 404);
      }

      $nid = $row->nid;
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      $data = [
        'nid' => $nid,
        'langcode' => $requested_langcode,
        'language_name' => $language_name,
        'phone' => $row->phone_no ?? '',
        'email' => $row->content_email ?? '',
        'address' => $row->address ?? '',
      ];

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data available in requested language.'
          : 'Contact information fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function contactBanner(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 235);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function contactformSubmit()
  {
    $request = \Drupal::request();
    $data = $request->request->all();

    // Validate and process the contact form submission
    if (empty($data['name']) || empty($data['email']) || empty($data['message'])) {
      return new JsonResponse([
        'status' => false,
        'message' => 'All fields are required.',
      ], 400);
    }

    // Here you would typically save the data or send an email
    // For demonstration, we'll just log the data
    \Drupal::logger('custom_module')->info('Contact form submitted: @data', ['@data' => json_encode($data)]);

    return new JsonResponse([
      'status' => true,
      'message' => 'Contact form submitted successfully.',
    ]);
  }
}
