<?php

namespace Drupal\api_module\Controller\AllDetailsPage;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;


class RegulatoryPageController extends ControllerBase
{
  public function getRegulatoryBannerdetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 17);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getRegulatoryCollagedetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_regulatory_collage_icon', 'nfrci', 'nfrci.entity_id = nfd.nid AND nfrci.delta = 0 AND nfrci.langcode = :langcode');
        $query->addField('nfrci', 'field_regulatory_collage_icon_target_id', 'image_icon');

        $query->leftJoin('node__field_reg_collage_text', 'nfrct', 'nfrct.entity_id = nfd.nid AND nfrct.delta = 0 AND nfrct.langcode = :langcode');
        $query->addField('nfrct', 'field_reg_collage_text_value', 'collage_text');

        $query->leftJoin('node__field_regulatory_collage_count', 'nfrccount', 'nfrccount.entity_id = nfd.nid AND nfrccount.delta = 0 AND nfrccount.langcode = :langcode');
        $query->addField('nfrccount', 'field_regulatory_collage_count_value', 'count_value');

        $query->condition('nfd.type', 'regulatory_section_collage');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $img_url = '';
        if (!empty($row->image_icon)) {
          $file = \Drupal\file\Entity\File::load($row->image_icon);
          if ($file) {
            $img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $row->nid,
          'collage_text' => strip_tags($row->collage_text ?? ''),
          'count_value' => strip_tags($row->count_value ?? ''),
          'image_icon' => $img_url,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Collage data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }

  public function getRegulatoryCountContentDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_reg_cont_description', 'nfrcdes', 'nfrcdes.entity_id = nfd.nid AND nfrcdes.delta = 0 AND nfrcdes.langcode = :langcode');
        $query->addField('nfrcdes', 'field_reg_cont_description_value', 'description_details');

        $query->leftJoin('node__field_reg_cont_heading', 'nfrchead', 'nfrchead.entity_id = nfd.nid AND nfrchead.delta = 0 AND nfrchead.langcode = :langcode');
        $query->addField('nfrchead', 'field_reg_cont_heading_value', 'content_heading');

        $query->condition('nfd.type', 'regulatory_content');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $data = [
          'nid' => $row->nid,
          'heading' => $row->content_heading ?? '',
          'description' => $row->description_details ?? '',
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Regulatory content details data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getRegulatoryGeographyContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        // Join field_country_name with language filter
        $query->leftJoin(
          'node__field_country_name',
          'nfcn',
          'nfcn.entity_id = nfd.nid AND nfcn.delta = 0 AND nfcn.langcode = :langcode'
        );
        $query->addField('nfcn', 'field_country_name_value', 'country_name');

        // Join field_filings with language filter
        $query->leftJoin(
          'node__field_filings',
          'nff',
          'nff.entity_id = nfd.nid AND nff.delta = 0 AND nff.langcode = :langcode'
        );
        $query->addField('nff', 'field_filings_value', 'country_filing');

        $query->condition('nfd.type', 'regulatory_geography');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->distinct();

        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        // Fallback to English if no results for requested language
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $data[] = [
          'nid' => $row->nid,
          'title' => $row->title ?? '',
          'country_name' => strip_tags($row->country_name) ?? '',
          'country_filing' => strip_tags($row->country_filing) ?? '',
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Country Geography data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getRegulatoryGeographyBannerContent(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin(
          'node__field_reg_geo_image',
          'nfabi',
          'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode'
        );
        $query->addField('nfabi', 'field_reg_geo_image_target_id', 'geo_banner_image');

        $query->condition('nfd.type', 'regulatory_geography_image');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);

        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->geo_banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->geo_banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'title' => $row->title ?? '',
          'geo_banner_image' => $bannerImg_url,
          'langcode' => $row->langcode,
          'language_name' => \Drupal::languageManager()->getStandardLanguageList()[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No banner data found in requested language.'
          : 'Geo Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getRegulatoryFillingupdateDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin(
          'node__field_slider_country',
          'nfsc',
          'nfsc.entity_id = nfd.nid AND nfsc.delta = 0 AND nfsc.langcode = :langcode'
        );
        $query->addField('nfsc', 'field_slider_country_value', 'country_name');

        $query->leftJoin(
          'node__field_slider_link',
          'nfslink',
          'nfslink.entity_id = nfd.nid AND nfslink.delta = 0 AND nfslink.langcode = :langcode'
        );
        $query->addField('nfslink', 'field_slider_link_value', 'content_link');

        $query->leftJoin(
          'node__field_slider_tex',
          'nfstex',
          'nfstex.entity_id = nfd.nid AND nfstex.delta = 0 AND nfstex.langcode = :langcode'
        );
        $query->addField('nfstex', 'field_slider_tex_value', 'content_text');

        $query->leftJoin(
          'node__field_year',
          'nfyear',
          'nfyear.entity_id = nfd.nid AND nfyear.delta = 0 AND nfyear.langcode = :langcode'
        );
        $query->addField('nfyear', 'field_year_value', 'content_year');

        $query->condition('nfd.type', 'regulatory_filing_slider');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $data[] = [
          'nid' => $row->nid,
          'country_name' => $row->country_name ?? '',
          'content_link' => $row->content_link ?? '',
          'content_text' => $row->content_text ?? '',
          'content_year' => $row->content_year ?? '',
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Regulatory country slider details data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }





  public function getRegulatoryGeoData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'regulatory_geography_description')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $updated_date = $node->get('field_last_updated_date')->value ?? "";
        $description = $node->get('field_geography_below_descriptio')->value ?? "";
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'updated_date' => $updated_date,
          'description' => $description,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }
}
