<?php

namespace Drupal\api_module\Controller\AllDetailsPage;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class RDPageController.
 *
 * Provides route responses for the AllDetailsPage.
 */
class RDPageController extends ControllerBase
{

  /**
   * Returns a simple render array for the details page.
   */

  public function getRdBannerdetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 12);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getRdContentDetails(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'r_d_sections')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        // Join the field table that stores the taxonomy term reference.
        $query->join('node__field_r_d_content_section_type', 'sec', 'nfd.nid = sec.entity_id');
        $query->condition('sec.field_r_d_content_section_type_target_id', 18);

        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }
      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $content_heading = $node->get('field_heading')->value ?? '';
        $content_description = $node->get('field_description')->value ?? '';
        $know_more_link = $node->get('field_link')->value ?? '';
        $main_image = $node->get('field_section_image')->target_id ?? '';
        $icon_image = $node->get('field_section_icon')->target_id ?? '';

        $main_img_url = '';
        $icon_img_url = '';
        if (!empty($main_image)) {
          $file = \Drupal\file\Entity\File::load($main_image);
          if ($file) {
            $main_img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        if (!empty($icon_image)) {
          $file = \Drupal\file\Entity\File::load($icon_image);
          if ($file) {
            $icon_img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        $data = [
          'nid' => $nid,
          'content_heading' => $content_heading,
          'content_description' => $content_description,
          'main_img_url' => $main_img_url,
          'icon_img_url' => $icon_img_url,
          'know_more_link' => $know_more_link ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'Content data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getRdTertiaryContentDetails(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'r_d_sections')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        // Join the field table that stores the taxonomy term reference.
        $query->join('node__field_r_d_content_section_type', 'sec', 'nfd.nid = sec.entity_id');
        $query->condition('sec.field_r_d_content_section_type_target_id', 20);

        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }
      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $content_heading = $node->get('field_heading')->value ?? '';
        $content_description = $node->get('field_description')->value ?? '';
        $know_more_link = $node->get('field_link')->value ?? '';
        $main_image = $node->get('field_section_image')->target_id ?? '';
        $icon_image = $node->get('field_section_icon')->target_id ?? '';

        $main_img_url = '';
        $icon_img_url = '';
        if (!empty($main_image)) {
          $file = \Drupal\file\Entity\File::load($main_image);
          if ($file) {
            $main_img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        if (!empty($icon_image)) {
          $file = \Drupal\file\Entity\File::load($icon_image);
          if ($file) {
            $icon_img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        $data = [
          'nid' => $nid,
          'content_heading' => $content_heading,
          'content_description' => $content_description,
          'main_img_url' => $main_img_url,
          'icon_img_url' => $icon_img_url,
          'know_more_link' => $know_more_link ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'Content data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getRdSecondaryContentDetails(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'r_d_sections')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);

        // Join the field table that stores the taxonomy term reference.
        $query->join('node__field_r_d_content_section_type', 'sec', 'nfd.nid = sec.entity_id');
        $query->condition('sec.field_r_d_content_section_type_target_id', 19);

        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }
      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        $content_heading = $node->get('field_heading')->value ?? '';
        $content_description = $node->get('field_description')->value ?? '';
        $know_more_link = $node->get('field_link')->value ?? '';
        $main_image = $node->get('field_section_image')->target_id ?? '';
        $icon_image = $node->get('field_section_icon')->target_id ?? '';

        $main_img_url = '';
        $icon_img_url = '';
        if (!empty($main_image)) {
          $file = \Drupal\file\Entity\File::load($main_image);
          if ($file) {
            $main_img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        if (!empty($icon_image)) {
          $file = \Drupal\file\Entity\File::load($icon_image);
          if ($file) {
            $icon_img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }
        $data[] = [
          'nid' => $nid,
          'content_heading' => $content_heading,
          'content_description' => $content_description,
          'main_img_url' => $main_img_url,
          'icon_img_url' => $icon_img_url,
          'know_more_link' => $know_more_link ?? '',
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'Content data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function getRdCapabilitiesContentDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchContent = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_capability_image', 'nfcapImg', 'nfcapImg.entity_id = nfd.nid AND nfcapImg.delta = 0 AND nfcapImg.langcode = :langcode');
        $query->addField('nfcapImg', 'field_capability_image_target_id', 'image_icon');

        $query->leftJoin('node__field_capability_text', 'nfcaptext', 'nfcaptext.entity_id = nfd.nid AND nfcaptext.delta = 0 AND nfcaptext.langcode = :langcode');
        $query->addField('nfcaptext', 'field_capability_text_value', 'capability_txt');

        $query->condition('nfd.type', 'r_d_capabilities');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Step 1: Try requested language
      $results = $fetchContent($requested_langcode);

      // Step 2: Fallback to English if needed
      $fallback = false;
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchContent('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $nid = $row->nid;
        $img_url = '';

        if (!empty($row->image_icon)) {
          $file = \Drupal\file\Entity\File::load($row->image_icon);
          if ($file) {
            $img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $nid,
          'capability_txt' => strip_tags($row->capability_txt ?? ''),
          'image_icon' => $img_url,
          'langcode' => $row->langcode,
          'language_name' => isset($lang_names[$row->langcode]) ? $lang_names[$row->langcode][0] : ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'Capability data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      \Drupal::logger('custom_module')->error($th->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getRdGenericFormulationContentDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      // Shared query logic
      $fetchContent = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_r_d_gfo_icon', 'nfgrdic', 'nfgrdic.entity_id = nfd.nid AND nfgrdic.delta = 0 AND nfgrdic.langcode = :langcode');
        $query->addField('nfgrdic', 'field_r_d_gfo_icon_target_id', 'image_icon');

        $query->leftJoin('node__field_r_d_gfo_text', 'nfrdtext', 'nfrdtext.entity_id = nfd.nid AND nfrdtext.delta = 0 AND nfrdtext.langcode = :langcode');
        $query->addField('nfrdtext', 'field_r_d_gfo_text_value', 'gfo_txt');

        $query->condition('nfd.type', 'r_d_generic_formulation_offering');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Step 1: Fetch in requested language
      $results = $fetchContent($requested_langcode);

      // Step 2: Fallback to English if empty
      $fallback = false;
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchContent('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $nid = $row->nid;

        $img_url = '';
        if (!empty($row->image_icon)) {
          $file = \Drupal\file\Entity\File::load($row->image_icon);
          if ($file) {
            $img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $nid,
          'gfo_txt' => strip_tags($row->gfo_txt ?? ''),
          'image_icon' => $img_url,
          'langcode' => $row->langcode,
          'language_name' => isset($lang_names[$row->langcode]) ? $lang_names[$row->langcode][0] : ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'R&D generic formulation data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      \Drupal::logger('custom_module')->error($th->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function getRdmodelscollaborationContentDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language Parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchContent = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_r_d_cm_icon', 'nfgrdic', 'nfgrdic.entity_id = nfd.nid AND nfgrdic.delta = 0 AND nfgrdic.langcode = :langcode');
        $query->addField('nfgrdic', 'field_r_d_cm_icon_target_id', 'image_icon');

        $query->leftJoin('node__field_r_d_cm_text', 'nfrdcmtext', 'nfrdcmtext.entity_id = nfd.nid AND nfrdcmtext.delta = 0 AND nfrdcmtext.langcode = :langcode');
        $query->addField('nfrdcmtext', 'field_r_d_cm_text_value', 'cm_text');

        $query->condition('nfd.type', 'r_d_collaboration_models');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Step 1: Try requested language
      $results = $fetchContent($requested_langcode);

      // Step 2: Fallback to English if nothing found
      $fallback = false;
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchContent('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $nid = $row->nid;
        $img_url = '';

        if (!empty($row->image_icon)) {
          $file = \Drupal\file\Entity\File::load($row->image_icon);
          if ($file) {
            $img_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $nid,
          'cm_text' => strip_tags($row->cm_text ?? ''),
          'image_icon' => $img_url,
          'langcode' => $row->langcode,
          'language_name' => isset($lang_names[$row->langcode]) ? $lang_names[$row->langcode][0] : ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data in requested language.'
          : 'Collaboration model content fetched successfully',
        'data' => $data,
      ]);
    } catch (\Throwable $th) {
      \Drupal::logger('custom_module')->error($th->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }
  public function getRdFaqData(Request $request)
  {
    try {
      $content = json_decode($request->getContent(), true);
      $requested_langcode = $content['lan'] ?? '';

      if (empty($requested_langcode)) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language code (lan) is required',
        ], 400);
      }

      $database = \Drupal::database();

      // Fetch node IDs for valid about_tab_year_content
      $fetchNodeIds = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid'])
          ->condition('nfd.type', 'r_d_faq_section')
          ->condition('nfd.status', 1)
          ->condition('nfd.langcode', $langcode);
        return $query->execute()->fetchCol();
      };

      $nids = $fetchNodeIds($requested_langcode);
      $fallback = false;

      // Fallback to English if no nodes found
      if (empty($nids) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $nids = $fetchNodeIds('en');
        $fallback = true;
      }

      $nodes = \Drupal\node\Entity\Node::loadMultiple($nids);
      $response = [];

      $language = \Drupal::languageManager()->getLanguage($requested_langcode);
      $language_name = $language ? $language->getName() : $requested_langcode;

      foreach ($nodes as $node) {
        if ($node->hasTranslation($requested_langcode)) {
          $node = $node->getTranslation($requested_langcode);
        }

        $nid = $node->id();
        if ($node->hasField('field_faqs') && !$node->get('field_faqs')->isEmpty()) {
          foreach ($node->get('field_faqs') as $para_ref) {
            $para = $para_ref->entity;
            if ($para && $para->hasTranslation($requested_langcode)) {
              $para = $para->getTranslation($requested_langcode);
            }

            $question = $para->get('field_question')->value ?? '';
            $answer = $para->get('field_answer')->value ?? '';

            $response[] = [
              'question' => $question,
              'answer' => $answer,
            ];
          }
        }
        $response = [
          'nid' => $nid,
          'langcode' => $requested_langcode,
          'language_name' => $language_name,
          'faqs' => $response,
        ];
      }
      return new JsonResponse([
        'status' => true,
        'message' => $fallback == true ? 'Fallback to English. As No Data found in the requested language.' : 'Data fetch successfully',
        'data' => $response,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred' . $e->getMessage(),
      ], 500);
    }
  }
}
