<?php


namespace Drupal\api_module\Controller\AllDetailsPage;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;


class ApiManufactureController extends ControllerBase
{
  public function getApiManufactureBannerDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'title', 'langcode']);

        $query->leftJoin('node__field_banner_heading', 'nfbh', 'nfbh.entity_id = nfd.nid AND nfbh.delta = 0 AND nfbh.langcode = :langcode');
        $query->addField('nfbh', 'field_banner_heading_value', 'banner_heading');

        $query->leftJoin('node__field_banner_image', 'nfabi', 'nfabi.entity_id = nfd.nid AND nfabi.delta = 0 AND nfabi.langcode = :langcode');
        $query->addField('nfabi', 'field_banner_image_target_id', 'banner_image');

        $query->leftJoin('node__field_common_banner_brochure', 'nfcbb', 'nfcbb.entity_id = nfd.nid AND nfcbb.delta = 0 AND nfcbb.langcode = :langcode');
        $query->addField('nfcbb', 'field_common_banner_brochure_target_id', 'brochure');

        $query->leftJoin('node__field_navigation_link', 'nfablin', 'nfablin.entity_id = nfd.nid AND nfablin.delta = 0 AND nfablin.langcode = :langcode');
        $query->addField('nfablin', 'field_navigation_link_value', 'banner_navigation_link');

        $query->leftJoin('node__field_page_type', 'nfptyp', 'nfptyp.entity_id = nfd.nid AND nfptyp.delta = 0');
        $query->addField('nfptyp', 'field_page_type_target_id', 'page_type');

        $query->leftJoin('taxonomy_term_field_data', 'ttfd', 'ttfd.tid = nfptyp.field_page_type_target_id');
        $query->addField('ttfd', 'name', 'page_type_name');

        $query->condition('nfd.type', 'common_inner_banner_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfptyp.field_page_type_target_id', 33);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $bannerImg_url = '';
        if (!empty($row->banner_image)) {
          $file = \Drupal\file\Entity\File::load($row->banner_image);
          if ($file) {
            $bannerImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $brochure = '';
        if (!empty($row->brochure)) {
          $brochureFile = \Drupal\file\Entity\File::load($row->brochure);
          if ($brochureFile) {
            $brochure = \Drupal::service('file_url_generator')->generateAbsoluteString($brochureFile->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'banner_navigation' => $row->banner_navigation_link ?? '',
          'banner_heading' => $row->banner_heading ?? '',
          'banner_image' => $bannerImg_url,
          'banner_download_brochure_link' => $brochure,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Banner data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }


  public function getApiManufactureMainContentDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        $query->leftJoin('node__field_manu_cap_main_section_desc', 'nfauc', 'nfauc.entity_id = nfd.nid AND nfauc.delta = 0 AND nfauc.langcode = :langcode');
        $query->addField('nfauc', 'field_manu_cap_main_section_desc_value', 'description');

        $query->leftJoin('node__field_manu_cap_main_section_head', 'nfauhead', 'nfauhead.entity_id = nfd.nid AND nfauhead.delta = 0 AND nfauhead.langcode = :langcode');
        $query->addField('nfauhead', 'field_manu_cap_main_section_head_value', 'heading');

        $query->leftJoin('node__field_manu_cap_main_section_icon', 'nfauci', 'nfauci.entity_id = nfd.nid AND nfauci.delta = 0 AND nfauci.langcode = :langcode');
        $query->addField('nfauci', 'field_manu_cap_main_section_icon_target_id', 'content_image_icon');

        $query->leftJoin('node__field_manu_cap_main_section_img', 'nfaucimage', 'nfaucimage.entity_id = nfd.nid AND nfaucimage.delta = 0 AND nfaucimage.langcode = :langcode');
        $query->addField('nfaucimage', 'field_manu_cap_main_section_img_target_id', 'content_main_image');

        $query->condition('nfd.type', 'manu_cap_main_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->range(0, 1);
        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Attempt to fetch in requested language
      $results = $fetchData($requested_langcode);

      // Fallback to English
      $fallback = false;
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $content_iconImg_url = '';
        if (!empty($row->content_image_icon)) {
          $file = \Drupal\file\Entity\File::load($row->content_image_icon);
          if ($file) {
            $content_iconImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $contentImg_url = '';
        if (!empty($row->content_main_image)) {
          $file = \Drupal\file\Entity\File::load($row->content_main_image);
          if ($file) {
            $contentImg_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'heading' => $row->heading ?? '',
          'about_content' => $row->description ?? '',
          'content_img' => $contentImg_url,
          'content_icon' => $content_iconImg_url,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api manufacture main content fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('custom_module')->error($e->getMessage());
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $e->getMessage(),
      ], 500);
    }
  }

  public function ApiManugetureGraffitiContentDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        $query->leftJoin('node__field_manu_cap_graffiti_desc', 'nfauoccd', 'nfauoccd.entity_id = nfd.nid AND nfauoccd.delta = 0 AND nfauoccd.langcode = :langcode');
        $query->addField('nfauoccd', 'field_manu_cap_graffiti_desc_value', 'description');

        $query->leftJoin('node__field_manu_cap_graffiti_heading', 'nfauocch', 'nfauocch.entity_id = nfd.nid AND nfauocch.delta = 0 AND nfauocch.langcode = :langcode');
        $query->addField('nfauocch', 'field_manu_cap_graffiti_heading_value', 'heading');

        $query->leftJoin('node__field__manu_cap_graffiti_icon_', 'nfauocci', 'nfauocci.entity_id = nfd.nid AND nfauocci.delta = 0 AND nfauocci.langcode = :langcode');
        $query->addField('nfauocci', 'field__manu_cap_graffiti_icon__target_id', 'icon');

        $query->condition('nfd.type', 'manu_cap_graffiti_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Try requested language first
      $results = $fetchData($requested_langcode);
      $fallback = false;

      // Fallback to English if no result
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $iconUrl = '';
        if (!empty($row->icon)) {
          $file = \Drupal\file\Entity\File::load($row->icon);
          if ($file) {
            $iconUrl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $row->nid,
          'description' => $row->description ?? '',
          'heading' => $row->heading ?? '',
          'contentIcon' => $iconUrl,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api Manufacture Graffiti data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function ApiManugetureLocationDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        $query->leftJoin('node__field__manu_cap_location_img_', 'nfauocci', 'nfauocci.entity_id = nfd.nid AND nfauocci.delta = 0 AND nfauocci.langcode = :langcode');
        $query->addField('nfauocci', 'field__manu_cap_location_img__target_id', 'loc_image');
        
        $query->leftJoin('node__field_location_heading', 'nflh', 'nflh.entity_id = nfd.nid AND nflh.delta = 0 AND nflh.langcode = :langcode');
        $query->addField('nflh', 'field_location_heading_value', 'heading');

        $query->condition('nfd.type', 'manu_cap_locations_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();

        return $query->execute()->fetchAll();
      };

      // Attempt fetch with requested language
      $results = $fetchData($requested_langcode);
      $fallback = false;

      // Fallback to English if needed
      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $imageurl = '';
        if (!empty($row->loc_image)) {
          $file = \Drupal\file\Entity\File::load($row->loc_image);
          if ($file) {
            $imageurl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'heading' => $row->heading,
          'location_image' => $imageurl,
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api Manufacture Location data fetched successfully',
        'lan_code' => $requested_langcode,
        'language' => $lang_names[$requested_langcode][0],
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function ApiManugetureCapabilitiesDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        $query->leftJoin('node__field_manu_cap_details_desc', 'nfauodes', 'nfauodes.entity_id = nfd.nid AND nfauodes.delta = 0 AND nfauodes.langcode = :langcode');
        $query->addField('nfauodes', 'field_manu_cap_details_desc_value', 'description');

        $query->leftJoin('node__field_manu_cap_details_heading', 'nfauocchh', 'nfauocchh.entity_id = nfd.nid AND nfauocchh.delta = 0 AND nfauocchh.langcode = :langcode');
        $query->addField('nfauocchh', 'field_manu_cap_details_heading_value', 'heading');

        $query->leftJoin('node__field_manu_cap_details_img', 'nfauocci', 'nfauocci.entity_id = nfd.nid AND nfauocci.delta = 0 AND nfauocci.langcode = :langcode');
        $query->addField('nfauocci', 'field_manu_cap_details_img_target_id', 'image');

        $query->condition('nfd.type', 'manu_cap_details_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();

        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $imgUrl = '';
        if (!empty($row->image)) {
          $file = \Drupal\file\Entity\File::load($row->image);
          if ($file) {
            $imgUrl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data = [
          'nid' => $row->nid,
          'description' => $row->description ?? '',
          'heading' => $row->heading ?? '',
          'contentIcon' => $imgUrl,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api Manufacture capabilities data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function ApiManugetureQualityDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        $query->leftJoin('node__field_manu_cap_qc_heading', 'nfauhead', 'nfauhead.entity_id = nfd.nid AND nfauhead.delta = 0 AND nfauhead.langcode = :langcode');
        $query->addField('nfauhead', 'field_manu_cap_qc_heading_value', 'heading');

        $query->leftJoin('node__field_manu_cap_qc_text', 'nfautext', 'nfautext.entity_id = nfd.nid AND nfautext.delta = 0 AND nfautext.langcode = :langcode');
        $query->addField('nfautext', 'field_manu_cap_qc_text_value', 'details');

        $query->leftJoin('node__field_manu_cap_qc_img', 'nfauocci', 'nfauocci.entity_id = nfd.nid AND nfauocci.delta = 0 AND nfauocci.langcode = :langcode');
        $query->addField('nfauocci', 'field_manu_cap_qc_img_target_id', 'image');

        $query->condition('nfd.type', 'manu_cap_quality_control');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $imgUrl = '';
        if (!empty($row->image)) {
          $file = \Drupal\file\Entity\File::load($row->image);
          if ($file) {
            $imgUrl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $row->nid,
          'heading' => $row->heading ?? '',
          'details' => $row->details ?? '',
          'image_icon' => $imgUrl,
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api Manufacture qualities data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }


  public function ApiManugetureRelatedPageDetails(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {
        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        $query->leftJoin('node__field_manu_cap_rel_page_heading', 'nfauocchh', 'nfauocchh.entity_id = nfd.nid AND nfauocchh.delta = 0 AND nfauocchh.langcode = :langcode');
        $query->addField('nfauocchh', 'field_manu_cap_rel_page_heading_value', 'page_heading');

        $query->leftJoin('node__field_manu_cap_rel_page_img', 'nfmcrpim', 'nfmcrpim.entity_id = nfd.nid AND nfmcrpim.delta = 0 AND nfmcrpim.langcode = :langcode');
        $query->addField('nfmcrpim', 'field_manu_cap_rel_page_img_target_id', 'page_background_image');

        $query->leftJoin('node__field_manu_cap_rel_page_link', 'nfcplink', 'nfcplink.entity_id = nfd.nid AND nfcplink.delta = 0 AND nfcplink.langcode = :langcode');
        $query->addField('nfcplink', 'field_manu_cap_rel_page_link_value', 'page_link');

        $query->condition('nfd.type', 'manu_cap_rel_pages');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);
        $query->addExpression(':langcode', 'langcode', [':langcode' => $langcode]);

        $query->distinct();
        return $query->execute()->fetchAll();
      };

      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {
        $relatedPageImg = '';
        if (!empty($row->page_background_image)) {
          $file = \Drupal\file\Entity\File::load($row->page_background_image);
          if ($file) {
            $relatedPageImg = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          }
        }

        $data[] = [
          'nid' => $row->nid,
          'related_page_heading' => $row->page_heading ?? '',
          'related_page_background_image' => $relatedPageImg,
          'related_page_link' => $row->page_link ?? '',
          'langcode' => $row->langcode,
          'language_name' => $lang_names[$row->langcode][0] ?? ucfirst($row->langcode),
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api Manufacture related page data fetched successfully',
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }
  public function ApiManugetureDetailsNumberData(Request $request)
  {
    try {
      $content = $request->getContent();
      $input = json_decode($content, TRUE);
      $requested_langcode = isset($input['lan']) ? $input['lan'] : 'en';

      if (empty($input['lan'])) {
        return new JsonResponse([
          'status' => false,
          'message' => 'Language parameter is missing',
        ], 400);
      }

      $database = \Drupal::database();

      $fetchData = function ($langcode) use ($database) {

        $query = $database->select('node_field_data', 'nfd')
          ->fields('nfd', ['nid', 'langcode']);

        // Filter only manu_cap_details_number_section nodes
        $query->condition('nfd.type', 'manu_cap_details_number_section');
        $query->condition('nfd.status', 1);
        $query->condition('nfd.langcode', $langcode);

        // ---- JOIN: field_capability_heading ----
        $query->leftJoin(
          'node__field_capability_heading',
          'fch',
          'fch.entity_id = nfd.nid AND fch.delta = 0 AND fch.langcode = :fch_lang'
        );
        $query->addField('fch', 'field_capability_heading_value', 'heading');

        // ---- JOIN: field_text ----
        $query->leftJoin(
          'node__field_text',
          'ft',
          'ft.entity_id = nfd.nid AND ft.delta = 0 AND ft.langcode = :ft_lang'
        );
        $query->addField('ft', 'field_text_value', 'text');

        // Bind both placeholders separately
        $query->addExpression(':fch_lang', 'fch_lang', [':fch_lang' => $langcode]);
        $query->addExpression(':ft_lang', 'ft_lang', [':ft_lang' => $langcode]);

        $query->distinct();

        return $query->execute()->fetchAll();
      };
      $results = $fetchData($requested_langcode);
      $fallback = false;

      if (empty($results) && $requested_langcode !== 'en') {
        $requested_langcode = 'en';
        $results = $fetchData('en');
        $fallback = true;
      }

      $lang_names = \Drupal::languageManager()->getStandardLanguageList();
      $data = [];

      foreach ($results as $row) {

        $data[] = [
          'nid' => $row->nid,
          'heading' => $row->heading ?? '',
          'text' => $row->text ?? ''
        ];
      }

      return new JsonResponse([
        'status' => true,
        'message' => $fallback
          ? 'Fallback to English. No data found in requested language.'
          : 'Api Manufacture Details Numbered Data Fetched successfully',
        'lan_code' => $requested_langcode,
        'language' => $lang_names[$requested_langcode][0],
        'data' => $data,
      ]);
    } catch (\Exception $th) {
      return new JsonResponse([
        'status' => false,
        'message' => 'An error occurred: ' . $th->getMessage(),
      ], 500);
    }
  }
}
