'use client';
import React, { useState } from 'react';
import { HiOutlineX, HiOutlineStar } from 'react-icons/hi';

interface FeedbackModalProps {
  isOpen: boolean;
  onClose: () => void;
}

const destinations = ['Paris', 'New York', 'Tokyo', 'Sydney', 'Cairo'];

const FeedbackModal: React.FC<FeedbackModalProps> = ({ isOpen, onClose }) => {
  const [form, setForm] = useState({
    name: '',
    email: '',
    mobile: '',
    destination: '',
    title: '',
    review: '',
  });

  const handleChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>
  ) => {
    const { name, value } = e.target;
    setForm((prev) => ({ ...prev, [name]: value }));
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    console.log(form);
    onClose();
  };

  if (!isOpen) return null;

  return (
    <div
      className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm"
      onClick={onClose}
      aria-modal="true"
      role="dialog"
      tabIndex={-1}
    >
      <div
        className="relative bg-white/10 backdrop-blur-md rounded-xl shadow-lg border border-white/20 max-w-lg w-full max-h-[85vh] overflow-y-auto p-6 text-white"
        onClick={(e) => e.stopPropagation()} // Prevent modal close on inner click
      >
        {/* Close button */}
        <button
          aria-label="Close modal"
          className="absolute top-4 right-4 text-white hover:text-red-400 text-2xl"
          onClick={onClose}
        >
          <HiOutlineX />
        </button>

        {/* Header */}
        <div className="flex flex-col items-center mb-6 space-y-1">
          <div className="bg-yellow-400/20 rounded-full p-3">
            <HiOutlineStar className="text-yellow-400 text-3xl" />
          </div>
          <h2 className="text-2xl font-semibold">We Value Your Feedback</h2>
          <p className="text-gray-300 text-center text-sm">
            Help us improve by sharing your experience
          </p>
        </div>

        {/* Form */}
        <form onSubmit={handleSubmit} className="space-y-5">
          {/** Input component with floating label style */}
          {[
            { id: 'name', label: 'Your Name', type: 'text' },
            { id: 'email', label: 'Email Address', type: 'email' },
            { id: 'mobile', label: 'Mobile Number', type: 'tel' },
          ].map(({ id, label, type }) => (
            <div key={id} className="relative">
              <input
                type={type}
                id={id}
                name={id}
                placeholder=" "
                value={form[id as keyof typeof form]}
                onChange={handleChange}
                required
                className="peer w-full bg-white/10 border border-white/20 rounded-lg px-4 pt-6 pb-2 text-white placeholder-transparent focus:outline-none focus:ring-2 focus:ring-blue-400 transition"
              />
              <label
                htmlFor={id}
                className="absolute left-4 top-2 text-sm text-gray-300 transition-all
                  peer-placeholder-shown:top-4
                  peer-placeholder-shown:text-base
                  peer-placeholder-shown:text-gray-400
                  peer-focus:top-2
                  peer-focus:text-sm
                  peer-focus:text-blue-300"
              >
                {label}
              </label>
            </div>
          ))}

          {/* Destination select */}
          <div className="relative">
            <select
              id="destination"
              name="destination"
              value={form.destination}
              onChange={handleChange}
              required
              className="peer w-full bg-white/10 border border-white/20 rounded-lg px-4 pt-6 pb-2 text-white appearance-none focus:outline-none focus:ring-2 focus:ring-blue-400"
            >
              <option value="" disabled hidden>
                Select Destination
              </option>
              {destinations.map((d) => (
                <option key={d} value={d} className="text-black">
                  {d}
                </option>
              ))}
            </select>
            <label
              htmlFor="destination"
              className="absolute left-4 top-2 text-sm text-gray-300 transition-all
                peer-placeholder-shown:top-4
                peer-placeholder-shown:text-base
                peer-placeholder-shown:text-gray-400
                peer-focus:top-2
                peer-focus:text-sm
                peer-focus:text-blue-300"
            >
              Destination
            </label>
          </div>

          {/* Review Title */}
          <div className="relative">
            <input
              type="text"
              id="title"
              name="title"
              placeholder=" "
              value={form.title}
              onChange={handleChange}
              required
              className="peer w-full bg-white/10 border border-white/20 rounded-lg px-4 pt-6 pb-2 text-white placeholder-transparent focus:outline-none focus:ring-2 focus:ring-blue-400 transition"
            />
            <label
              htmlFor="title"
              className="absolute left-4 top-2 text-sm text-gray-300 transition-all
                peer-placeholder-shown:top-4
                peer-placeholder-shown:text-base
                peer-placeholder-shown:text-gray-400
                peer-focus:top-2
                peer-focus:text-sm
                peer-focus:text-blue-300"
            >
              Review Title
            </label>
          </div>

          {/* Review Textarea */}
          <div className="relative">
            <textarea
              id="review"
              name="review"
              rows={4}
              placeholder=" "
              value={form.review}
              onChange={handleChange}
              required
              className="peer w-full bg-white/10 border border-white/20 rounded-lg px-4 pt-6 pb-2 text-white placeholder-transparent resize-none focus:outline-none focus:ring-2 focus:ring-blue-400 transition"
            />
            <label
              htmlFor="review"
              className="absolute left-4 top-2 text-sm text-gray-300 transition-all
                peer-placeholder-shown:top-4
                peer-placeholder-shown:text-base
                peer-placeholder-shown:text-gray-400
                peer-focus:top-2
                peer-focus:text-sm
                peer-focus:text-blue-300"
            >
              Write your review here...
            </label>
          </div>

          {/* Submit button */}
          <button
            type="submit"
            className="w-full py-3 bg-gradient-to-r from-blue-500 via-purple-500 to-indigo-500 rounded-lg font-semibold hover:opacity-90 transition"
          >
            Submit Review
          </button>
        </form>
      </div>
    </div>
  );
};

export default FeedbackModal;
