"use client";

import { Tabs, TabList, Tab, TabPanel } from "react-tabs";
import "react-tabs/style/react-tabs.css";
import Image from "next/image";
import Link from "next/link";
import AOS from "aos";
import "aos/dist/aos.css";
import { useEffect, useState } from "react";
import axios from "axios";
import { DEV_URL, XPublicToken } from "../../urls/apiUrls";

const ImageComponent = ({ src, alt, href }: any) => (
  <Link href={href || "#"} className="group block">
    <div
      className="position-relative overflow-hidden rounded-4 shadow-md transition-all duration-300 group-hover:shadow-xl group-hover:-translate-y-1"
      data-aos="zoom-in"
      data-aos-delay="100"
    >
      <Image
        src={src}
        alt={alt}
        width={400}
        height={300}
        className="img-fluid transition-transform duration-500 group-hover:scale-110 group-hover:brightness-95"
      />
      <div className="absolute inset-x-0 bottom-0 h-0 bg-gradient-to-t from-black/70 to-transparent transition-all duration-400 group-hover:h-2/3" />
    </div>
  </Link>
);

const TabWithImages: React.FC<any> = ({ internationalData }: any) => {
  const [countryData, setCountryData] = useState<any[]>([]);
  const [pagination, setPagination] = useState<any>({});
  const [selectedIndex, setSelectedIndex] = useState<number>(0);
  const [continentSlug, setContinentSlug] = useState<string | null>(null);
  const [page, setPage] = useState<number>(1);
  const [isLoading, setIsLoading] = useState<boolean>(false);

  // AOS init
  useEffect(() => {
    AOS.init({
      duration: 800,
      easing: "ease-in-out",
      once: true,
    });
  }, []);

  useEffect(() => {
    fetchCountries(continentSlug, page);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [continentSlug, page]);

  const fetchCountries = async (slug: string | null, page: number) => {
    try {
      setIsLoading(true);

      const url = slug
        ? `${DEV_URL}/api/v1/packages/countries/international?continent=${slug}&page=${page}`
        : `${DEV_URL}/api/v1/packages/countries/international?page=${page}`;

      const response = await axios.get(url, {
        headers: {
          "X-Public-Token": XPublicToken,
        },
      });

      const newData = response.data.data;
      const newCountries = newData.countries || [];

      setCountryData((prev: any[]) =>
        page === 1 ? newCountries : [...prev, ...newCountries]
      );
      setPagination(newData.pagination || {});
    } catch (err) {
      console.error("Failed to fetch:", err);
      setCountryData([]);
    } finally {
      setTimeout(() => {
        setIsLoading(false);
      }, 200);
    }
  };

  const handleTabSelect = (index: number) => {
    setSelectedIndex(index);
    setPage(1);
    const slug =
      index === 0 ? null : internationalData.continents[index - 1]?.slug;
    setContinentSlug(slug);
    setCountryData([]); // reset data on tab change
  };

  const handleLoadMore = () => {
    setPage((prev) => prev + 1);
  };

  return (
    <div className="container py-5">
      <div className="row">
        <Tabs selectedIndex={selectedIndex} onSelect={handleTabSelect}>
          <TabList className="nav nav-pills justify-content-center mb-5 gap-4">
            <Tab
              className="nav-link"
              selectedClassName="active bg-primary text-white"
            >
              All
            </Tab>
            {internationalData.continents.map((continent: any) => (
              <Tab
                key={continent.slug}
                className="nav-link"
                selectedClassName="active bg-primary text-white"
              >
                {continent.name}
              </Tab>
            ))}
          </TabList>

          {/* Panels */}
          {[null, ...internationalData.continents.map((c: any) => c.slug)].map(
            (slug, index) => (
              <TabPanel key={slug || "all"}>
                {selectedIndex === index && (
                  <>
                    <div className="row g-4">
                      {isLoading ? (
                        <div className="text-center w-100 py-4 text-gray-500">
                          Loading...
                        </div>
                      ) : countryData?.length > 0 ? (
                        countryData.map((data: any) => (
                          <div className="col-md-4" key={data.slug}>
                            <ImageComponent
                              src={data?.details?.banner_image}
                              alt={data.name}
                              href={`/international-holidays/${data.slug}`}
                            />
                            <Link
                              className="text-black"
                              href={`/international-holidays/${data.slug}`}
                            >
                              <h5 className="mt-4 text-center transition-colors duration-300">
                                {data.name}
                              </h5>
                            </Link>
                          </div>
                        ))
                      ) : (
                        <div className="text-center w-100 py-4 text-gray-500">
                          No destinations found.
                        </div>
                      )}
                    </div>

                    

                    {/* Load More */}
                    {!isLoading &&
                      pagination?.total > 0 &&
                      countryData.length < pagination.total && (
                        <div className="text-center mt-5">
                          <button
                            onClick={handleLoadMore}
                            className="btn orange-btn inline-flex items-center gap-2"
                          >
                            Load More
                            <span>
                              <Image
                                width={23}
                                height={23}
                                sizes="100vw"
                                src="/images/button-arrow.png"
                                alt=""
                              />
                            </span>
                          </button>
                        </div>
                      )}
                  </>
                )}
              </TabPanel>
            )
          )}
        </Tabs>
      </div>
    </div>
  );
};

export default TabWithImages;
