'use client';
import React, { useState, useEffect } from 'react';
import Image from 'next/image';
import Lightbox from 'yet-another-react-lightbox';
import 'yet-another-react-lightbox/styles.css';

interface TourDetailsBannerProps {
  image: string;
  images: { image_path: string }[];
  className?: string;
}

const TourDetailsBanner: React.FC<TourDetailsBannerProps> = ({ image, images, className }) => {
  const [photoIndex, setPhotoIndex] = useState(0);
  const [isOpen, setIsOpen] = useState(false);
  const [preloadedImages, setPreloadedImages] = useState<Set<string>>(new Set());
  const [isLoading, setIsLoading] = useState(false);

  // Combine the main image and the images array for the lightbox
  const allImages = [image, ...images.map((img) => img.image_path)];
  const slides = allImages.map((src) => ({ src }));

  const openLightbox = (index: number) => {
    setIsLoading(true);
    setPhotoIndex(index);

    if (preloadedImages.has(allImages[index])) {
      setIsOpen(true);
      setIsLoading(false);
    } else {
      const img = new window.Image();
      img.onload = () => {
        setIsOpen(true);
        setIsLoading(false);
        setPreloadedImages((prev) => new Set([...prev, allImages[index]]));
      };
      img.onerror = () => {
        setIsLoading(false);
      };
      img.src = allImages[index];
    }
  };

  return (
    <>
      <div className={`row position-relative ${className ?? ''}`}>
        {/* Left side: One large image */}
        <div className="col-lg-6 d-flex align-items-center justify-content-center group" onClick={() => openLightbox(0)}>
          <div className="position-relative w-100 h-100">
            <Image
              src={image}
              alt="Large left image"
              width={800}
              height={500}
              className="object-fit-cover w-100 h-100 transition-transform duration-300 custom-hover rounded"
              priority
            />
          </div>

          <div className="absolute inset-0 transition-colors duration-300" />
          {isLoading && (
            <div className="absolute inset-0 bg-black/20 flex items-center justify-center">
              <div className="animate-spin rounded-full h-8 w-8 border-2 border-white border-t-transparent"></div>
            </div>
          )}
        </div>

        {/* Right side: Two images on top, two on bottom */}
        <div className="col-lg-6">
          <div className="row mb-4">
            {/* Top row: Two images */}
            {images?.slice(0, 2).map((img, index) => (
              <div
                key={index}
                className="col-lg-6 d-flex align-items-center justify-content-center group"
                onClick={() => openLightbox(index + 1)}
              >
                <div className="position-relative w-100 h-100">
                  <Image
                    src={img.image_path}
                    alt={`Right top image ${index + 1}`}
                    width={400}
                    height={250}
                    className="object-fit-cover w-100 h-100 rounded transition-transform duration-300 custom-hover"
                  />
                  <div className="absolute inset-0 transition-colors duration-300" />
                  {isLoading && (
                    <div className="absolute inset-0 bg-black/20 flex items-center justify-center">
                      <div className="animate-spin rounded-full h-6 w-6 border-2 border-white border-t-transparent"></div>
                    </div>
                  )}
                </div>
              </div>
            ))}
          </div>
          <div className="row">
            {/* Bottom row: Two images */}
            {images?.slice(2, 4).map((img, index) => (
              <div
                key={index}
                className="col-lg-6 d-flex align-items-center justify-content-center group"
                onClick={() => openLightbox(index + 3)}
              >
                <div className="position-relative w-100 h-100">
                  <Image
                    src={img.image_path}
                    alt={`Right bottom image ${index + 1}`}
                    width={400}
                    height={250}
                    className="object-fit-cover w-100 h-100 rounded transition-transform duration-300 custom-hover"
                  />
                  <div className="absolute inset-0 transition-colors duration-300" />
                  {isLoading && (
                    <div className="absolute inset-0 bg-black/20 flex items-center justify-center">
                      <div className="animate-spin rounded-full h-6 w-6 border-2 border-white border-t-transparent"></div>
                    </div>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      <Lightbox
        open={isOpen}
        close={() => setIsOpen(false)}
        slides={slides}
        index={photoIndex}
        on={{
          view: ({ index }) => setPhotoIndex(index),
        }}
      />
    </>
  );
};

export default TourDetailsBanner;